<?php

namespace App\Http\Controllers\Admin;

use App\Http\Requests\User\LoginRequest;
use App\Http\Resources\Admin\UserListingResource;
use App\Models\Banner;
use App\Models\Order;
use Carbon\Carbon;
use Illuminate\Http\Request;
use App\Services\AdminService;
use Illuminate\Support\Facades\DB;
use App\Http\Resources\DefaultResource;
use App\Http\Controllers\MainApiController;
use App\Http\Requests\VerifyActivationCodeRequest;
use App\Http\Requests\Password\{ResetPassword, SetNewPasswordRequest, ChangePasswordRequest};
use App\Http\Requests\UpdateAdminProfileRequest;
use App\Models\User;
use Illuminate\Support\Facades\Auth;

class AuthController extends MainApiController
{
    private $adminService;

    public function __construct(AdminService $adminService)
    {
        parent::__construct();
        $this->adminService = $adminService;
    }


    public function update(UpdateAdminProfileRequest $request)
    {
        auth()->user()->first_name = $request->first_name;
        auth()->user()->last_name = $request->last_name;
        auth()->user()->phone = $request->phone_number;
        if ($request->has('image')) {
            $saveFile = saveFile($request->image, User::IMAGES_DIRECTORY, 'local');
            auth()->user()->image = $saveFile['fileName'];
        }
        auth()->user()->save();
        return $this->response->success(
            new DefaultResource(['message' => 'Password changed successfully!', 'data' => new UserListingResource(auth()->user())])
        );
        // return response()->json(['status' => true, 'message' => 'Profile Updated Successfully', 'data' => auth()->user()]);
    }


    public function updatePassword(ChangePasswordRequest $request)
    {
        $user = $this->adminService->updatePassword($request, auth()->user()->id);

        return $this->response->success(
            new DefaultResource(['message' => 'Password changed successfully!'])
        );
    }



    public function login(LoginRequest $request)
    {
        $data = $this->adminService->checkLogin($request, 'admin');
        return $this->response->success(
            new DefaultResource($data)
        );
    }

    public function reset(ResetPassword $request)
    {
        $response = $this->adminService->reset($request);
        if ($response) {
            return $this->response->success(
                new DefaultResource(['message' => 'We have sent you a verification code!'])
            );
        }
    }

    public function setPassword(SetNewPasswordRequest $request)
    {
        $data = $this->adminService->verifyAndUpdatePassword($request);
        return $this->response->success(
            new DefaultResource($data)
        );
    }

    public function verifyCode(VerifyActivationCodeRequest $request)
    {
        return $this->response->success(
            new DefaultResource(['message' => 'Code verified!'])
        );
    }

    public function sendVerificationCode(ResetPassword $request)
    {

        $user = $this->adminService->getUserByEmail($request->email);
        $this->adminService->sendAccountVerification($user);
        return $this->response->success(
            new DefaultResource(['message' => 'We have sent you a verification code!'])
        );
    }

    public function logout(Request $request)
    {
        $user = Auth::user();
        $user->tokens()->where('id', $user->currentAccessToken()->id)->delete();
        return $this->response->success(
            new DefaultResource(['message' => 'Logout Successful!'])
        );
    }

    public function profile(Request $request)
    {
        $admin = auth()->guard()->user();
        return $this->response->success(
            new DefaultResource($admin)
        );
    }

    public function dashboardStats(Request $request)
    {

        $monthlyTrend = User::selectRaw('COUNT(id) as count')->whereRaw("created_at LIKE '" . date('Y-m') . "%'")->where('type', 'user')->first();
        $totalUsers = User::where('type', 'user')->count();
        $totalBanners = Banner::count();
        $totalOrders = Order::where('is_paid', 1)->count();
        $monthlyTrend = (isset($monthlyTrend->count) && $monthlyTrend->count > 0 ? round((($monthlyTrend->count / $totalUsers) * 100), 2) : 0);
        $monthlyTrendApplication = User::selectRaw('COUNT(id) as count')->whereRaw("created_at LIKE '" . date('Y-m') . "%'")->first();
        $totalApplications = User::count();
        $monthlyTrendApplication = (isset($monthlyTrendApplication->count) && $monthlyTrendApplication->count > 0 ? round((($monthlyTrendApplication->count / $totalApplications) * 100), 2) : 0);
        $currentYear = request("year");
        // Prepare an array to hold the results
        $monthsCountUsers = array_fill(0, 12, 0); // Initialize an array with 12 zeros for each month
        $monthsCountEarnings = array_fill(0, 12, 0); // Initialize an array with 12 zeros for each month

        // Loop through each month from January to the current month
        for ($month = 1; $month <= Carbon::now()->month; $month++) {
            // Calculate the start and end dates for the current month
            $startOfMonth = Carbon::create($currentYear, $month, 1)->startOfMonth();
            $endOfMonth = Carbon::create($currentYear, $month, 1)->endOfMonth();

            // Count the number of users created in this month
            $count = User::where("type", 'user')->whereBetween('created_at', [$startOfMonth, $endOfMonth])->count();

            // Store the count in the corresponding month index (month - 1 because array index starts at 0)
            $monthsCountUsers[$month - 1] = $count;
        }
        for ($month = 1; $month <= Carbon::now()->month; $month++) {
            // Calculate the start and end dates for the current month
            $startOfMonth = Carbon::create($currentYear, $month, 1)->startOfMonth();
            $endOfMonth = Carbon::create($currentYear, $month, 1)->endOfMonth();

            // Fetch total earnings from commissions for the current month
            $totalEarnings = Order::where('is_paid', 1)->whereBetween('created_at', [$startOfMonth, $endOfMonth])
                ->get()
                ->sum(function ($order) {
                    // Calculate earnings: total_price * (commission / 100)
                    return $order->total_price;
                });

            // Store the total earnings in the corresponding month index
            $monthsCountEarnings[$month - 1] = $totalEarnings;
        }
        $totalEarnings = Order::where('is_paid', 1)->get()
            ->sum(function ($order) {
                // Calculate earnings: total_price * (commission / 100)
                return $order->total_price;
            });
        $data = [
            'status' => true,
            'total_users' => $totalUsers,
            'total_orders' => $totalOrders,
            'total_banners' => $totalBanners,
            'total_earnings' => round($totalEarnings, 2),
            'user_trend' => $monthlyTrend,
            'total_application' => $totalApplications,
            'application_trend' => $monthlyTrendApplication,
            'monthsCount' => $monthsCountUsers,
            'monthsCountEarnings' => $monthsCountEarnings,
        ];
        return $this->response->success(
            new DefaultResource(['data' => $data])
        );
    }

    public function chartStats(Request $request)
    {
        $currentYear = request("year");
        // Prepare an array to hold the results
        $monthsCountEarnings = array_fill(0, 12, 0); // Initialize an array with 12 zeros for each month

        // Loop through each month from January to the current month
        for ($month = 1; $month <= Carbon::now()->month; $month++) {
            // Calculate the start and end dates for the current month
            $startOfMonth = Carbon::create($currentYear, $month, 1)->startOfMonth();
            $endOfMonth = Carbon::create($currentYear, $month, 1)->endOfMonth();

            // Fetch total earnings from commissions for the current month
            $totalEarnings = Order::where('is_paid', 1)->whereBetween('created_at', [$startOfMonth, $endOfMonth])
                ->get()
                ->sum(function ($order) {
                    // Calculate earnings: total_price * (commission / 100)
                    return $order->total_price;
                });

            // Store the total earnings in the corresponding month index
            $monthsCountEarnings[$month - 1] = $totalEarnings;
        }
        $data = [
            'status' => true,
            'monthsCountEarnings' => $monthsCountEarnings,
        ];
        return $this->response->success(
            new DefaultResource(['data' => $data])
        );
    }

    public function userStats(Request $request)
    {
        $currentYear = request("year");
        // Prepare an array to hold the results
        $monthsCountUsers = array_fill(0, 12, 0); // Initialize an array with 12 zeros for each month

        // Loop through each month from January to the current month
        for ($month = 1; $month <= Carbon::now()->month; $month++) {
            // Calculate the start and end dates for the current month
            $startOfMonth = Carbon::create($currentYear, $month, 1)->startOfMonth();
            $endOfMonth = Carbon::create($currentYear, $month, 1)->endOfMonth();

            // Count the number of users created in this month
            $count = User::where("type", 'user')->whereBetween('created_at', [$startOfMonth, $endOfMonth])->count();

            // Store the count in the corresponding month index (month - 1 because array index starts at 0)
            $monthsCountUsers[$month - 1] = $count;
        }
        $data = [
            'status' => true,

            'monthsCount' => $monthsCountUsers,

        ];
        return $this->response->success(
            new DefaultResource(['data' => $data])
        );
    }

    public function chartData(Request $request)
    {
        $data = User::select(
            DB::raw("(COUNT(*)) as count"),
            DB::raw("MONTHNAME(created_at) as month_name")
        )
            ->whereYear('created_at', date('Y'))
            ->groupBy('month_name')
            ->get();
        $months = $records = [];
        foreach ($data as $dt) {
            $months[] = $dt['month_name'];
            $records[] = $dt['count'];
        }


        return response()->json(['status' => true, 'data' => $data, 'months' => $months, 'records' => $records]);
    }

    public function filterChartData(Request $request)
    {

        $filter = request()->input('filter');
        $dataQuery = User::select(
            DB::raw("(COUNT(*)) as count"),
            DB::raw("MONTHNAME(created_at) as month_name")
        )
            ->groupBy('month_name');

        if ($filter == "Y") {
            $data = $dataQuery->whereYear('created_at', date('Y'))->get();
        } elseif ($filter == "M") {
            $data = $dataQuery->whereMonth('created_at', date('m'))->get();
        } elseif ($filter == "Q") {
            $data = $dataQuery->whereBetween('created_at', [Carbon::now()->subMonths(3), Carbon::now()->endOfMonth()])->get();
        }

        $months = $records = [];
        foreach ($data as $dt) {
            $months[] = $dt['month_name'];
            $records[] = $dt['count'];
        }


        return response()->json(['status' => true, 'data' => $data, 'months' => $months, 'records' => $records]);
    }


}
