<?php

namespace App\Services;

use App\Exceptions\BadRequestException;
use App\Models\Chat;
use App\Models\ChatMessage;
use App\Models\ChatUser;
use App\Models\Friend;
use App\Models\FriendRequest;
use App\Models\User;
use App\Services\Common\BaseService;
use Exception;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\DB;
/**
 * Class JsonResponseService
 * @package App\Services
 */
class ChatService extends BaseService
{

    public function get_unread_chats()
    {
        $count = Chat::with('users', 'messages')
            ->whereHas('users', function ($q) {
                $q->where('user_id', auth()->id());
            })
            ->whereHas('messages', function ($q) {
                $q->whereNull('read_at')->where('user_id', '!=', auth()->id());
            })
            ->count();

        return $count;
    }

    public function createChat($request)
    {
        DB::beginTransaction();
        $user2ChatIds = ChatUser::where('user_id', $request->user_id)
            ->pluck('chat_id');

        $user3ChatIds = ChatUser::where('user_id', auth()->user()->id)
            ->pluck('chat_id');

        // Step 2: Find common chat_ids
        $commonChatIds = $user2ChatIds->intersect($user3ChatIds);

        // Step 3: Get private chat_ids from Chat model
        $privateChatIds = Chat::whereIn('id', $commonChatIds)
            ->where('type', 'private')
            ->pluck('id');

        // Step 4: Get records from ChatUser for these private chat_ids
        $record = ChatUser::whereIn('chat_id', $privateChatIds)
            ->whereIn('user_id', [$request->user_id, auth()->user()->id]) // Ensure it includes the specific user IDs
            ->first();
        if (!$record) {
            $chat = Chat::create([
                'type' => Chat::PRIVATE ,
            ]);

            $chat->users()->attach(auth()->id(), [
                'is_admin' => false,
                'created_at' => now(),
                'updated_at' => now()
            ]);
            $chat->users()->attach($request->user_id, [
                'is_admin' => false,
                'created_at' => now(),
                'updated_at' => now()
            ]);
            DB::commit();

            return $chat;
        }

        return Chat::find($record->chat_id);
    }

    public function createChatWithRequestAccept($sender, $receiver)
    {
        DB::beginTransaction();
        $user2ChatIds = ChatUser::where('user_id', $receiver)
            ->pluck('chat_id');

        $user3ChatIds = ChatUser::where('user_id', $sender)
            ->pluck('chat_id');

        // Step 2: Find common chat_ids
        $commonChatIds = $user2ChatIds->intersect($user3ChatIds);

        // Step 3: Get private chat_ids from Chat model
        $privateChatIds = Chat::whereIn('id', $commonChatIds)
            ->where('type', 'private')
            ->pluck('id');

        // Step 4: Get records from ChatUser for these private chat_ids
        $record = ChatUser::whereIn('chat_id', $privateChatIds)
            ->whereIn('user_id', [$receiver, $sender]) // Ensure it includes the specific user IDs
            ->first();
        if (!$record) {
            $chat = Chat::create([
                'type' => Chat::PRIVATE ,
            ]);

            $chat->users()->attach($sender, [
                'is_admin' => false,
                'created_at' => now(),
                'updated_at' => now()
            ]);
            $chat->users()->attach($receiver, [
                'is_admin' => false,
                'created_at' => now(),
                'updated_at' => now()
            ]);
            DB::commit();

            return $chat;
        }

        return true;
    }

    public function createGroup($request)
    {
        if ($request->has('image')) {
            $saveFile = saveFile($request->image, \HelperConstants::CHAT_IMAGE_DIRECTORY, \HelperConstants::UPLOAD_DISK);
            $image = $saveFile['fileName'];
        }
        DB::beginTransaction();
        $chat = Chat::create([
            'type' => Chat::GROUP,
            'name' => $request->name,
            'image' => $image ?? null,
        ]);

        $chat->users()->attach(auth()->id(), [
            'is_admin' => true,
            'created_at' => now(),
            'updated_at' => now()
        ]);

        foreach ($request->users as $user) {
            $chat->users()->attach($user, [
                'is_admin' => false,
                'created_at' => now(),
                'updated_at' => now()
            ]);
        }
        DB::commit();

        return $chat;
    }

    public function leaveGroup($id)
    {
        $chat = Chat::whereId($id)->whereHas('users', function ($q) {
            $q->where('user_id', auth()->id());
        })->first();

        if (!$chat || $chat->type != Chat::GROUP)
            throw new BadRequestException('Invalid Chat ID.');


        $is_admin = $chat->users()->where('user_id', auth()->id())->where('is_admin', true)->first();

        $chat->users()->detach(auth()->id());
        if ($chat->users()->count() == 0) {
            $chat->delete();

        } else {
            if ($is_admin) {
                $new_admin = $chat->users()->first();
                $chat->users()->where('user_id', $new_admin->id)->update(['is_admin' => true]);
            }
        }
        return "Group Left";
    }

    public function get_chat_list($request)
    {
        $searchTerms = explode(' ', request("search"));
        $chats = Chat::with('users', 'messages')
            ->whereHas('users', function ($q) {
                $q->where('user_id', auth()->id());
            })
            ->when($request->filled('search'), function ($q) use ($request, $searchTerms) {
                $q->where(function ($query) use ($request, $searchTerms) {
                    $query->where(function ($subquery) use ($request, $searchTerms) {
                        $subquery->where('type', Chat::PRIVATE )
                            ->whereHas('users', function ($subquery2) use ($request, $searchTerms) {
                                $subquery2->where(function ($q) use ($searchTerms) {
                                    foreach ($searchTerms as $term) {
                                        $q->where(function ($subQuery) use ($term) {
                                            $subQuery->where('first_name', 'like', '%' . $term . '%')
                                                ->orWhere('last_name', 'like', '%' . $term . '%')
                                                ->orWhere('email', 'like', '%' . $term . '%');
                                        });
                                    }
                                });
                            });
                    })
                        ->orWhere('name', 'like', '%' . $request->search . '%');
                });
            })
            ->when($request->filled('type'), function ($q) use ($request) {
                $q->where('type', $request->type);
            })
            ->orderByDesc(function ($q) {
                $q->select('created_at')
                    ->from('chat_messages')
                    ->whereColumn('chat_id', 'chats.id')
                    ->orderBy('created_at', 'desc')
                    ->limit(1);
            })
            // ->orderBy('created_at', 'desc')
            ->get();
        return $chats;
    }

    public function deleteChat($sender, $receiver)
    {
        $user2ChatIds = ChatUser::where('user_id', $receiver)->pluck('chat_id');
        $user3ChatIds = ChatUser::where('user_id', $sender)->pluck('chat_id');

        // Step 2: Find common chat_ids
        $commonChatIds = $user2ChatIds->intersect($user3ChatIds);

        // Step 3: Get private chat_ids from Chat model
        $privateChatIds = Chat::whereIn('id', $commonChatIds)
            ->where('type', 'private')
            ->pluck('id');

        // Step 4: Check if any private chat exists for the users
        if ($privateChatIds->isNotEmpty()) {
            // Get the first private chat ID
            $chatIdToDelete = $privateChatIds->first();

            // Delete chat record from Chat model
            $chat = Chat::find($chatIdToDelete);

            if ($chat) {
                // Proceed with deletion of the chat record
                $chat->delete();

                // Optionally, delete related records in ChatUser table (if needed)
                ChatUser::where('chat_id', $chatIdToDelete)->delete();

                DB::commit();
                return "Chat successfully deleted.";
            } else {
                throw new Exception('Chat not found.');
            }
        } else {
            throw new Exception('No private chat found between the users.');
        }
    }

    public function get_chat($id)
    {
        $chat = Chat::with('users', 'messages')
            ->whereHas('users', function ($q) {
                $q->where('user_id', auth()->id());
            })
            ->whereId($id)->first();

        if (!$chat)
            throw new BadRequestException('Invalid Chat ID.');

        $chat->messages()->whereNull('read_at')->where('user_id', '!=', auth()->id())->update(['read_at' => now()]);

        $messages = $chat->messages()->latest()->get();

        return $messages;
    }

    public function send_message($request)
    {
        $chat = Chat::with('users')->whereHas('users', function ($q) {
            $q->where('user_id', auth()->id());
        })->whereId($request->chat_id)->first();

        if (!$chat)
            throw new BadRequestException('Invalid Chat ID.');

        $chat->messages()->whereNull('read_at')->where('user_id', '!=', auth()->id())->update(['read_at' => now()]);

        $message = $chat->messages()->create([
            'user_id' => auth()->id(),
            'message' => $request->message,
        ]);
        if ($request->hasFile('file')) {
            storeImage($request, 'file', ChatMessage::IMAGES_DIRECTORY, $message);
        }
        return ChatMessage::find($message->id);
    }
}
