<?php


namespace App\Services\Common;

use App\Http\Resources\Admin\UserListingResource;
use App\Http\Resources\User\UserProfileWithAllDataResource;
use App\Models\TapToMatch;
use App\Models\UserLocation;
use App\Services\StripeService;
use Carbon\Carbon;
use Exception;
use Hash;
use DB;
use App\Models\User;
use HelperConstants;
use App\Models\PasswordReset;
use Illuminate\Auth\AuthenticationException;
use Illuminate\Validation\ValidationException;
use App\Notifications\{SignupActivation, PasswordResetRequest};
use App\Notifications\SendAdminNotification;


class UserService extends CrudService
{

    // use UploadAble;


    public function create(object $request)
    {
        $stripeService = new StripeService();
        $customer = $stripeService->createCustomer($request->first_name . " " . $request->last_name, $request->email);
        $user = new User();
        $user->first_name = $request->first_name;
        $user->last_name = $request->last_name;
        $user->stripe_customer_id = $customer->id;
        $user->email = $request->email;
        $user->email_verified_at = Carbon::now();
        $user->password = Hash::make($request->password);
        $user->dial_code = $request->dial_code;
        $user->phone = $request->phone;
        $user->date_of_birth = $request->date_of_birth;
        $user->age = $request->age;
        $user->country_code = $request->country_code;
        $user->about = $request->about;
        $user->device_id = $request->device_id;
        $user->nfc_tag = $request->nfc_tag;
        $user->gender = $request->gender;
        $user->height = $request->height;
        $user->education_level = $request->education_level;
        $user->live_location = 0;
        $user->career_field = $request->career_field;
        $user->device_token = $request->device_token;
        $user->avatar_id = $request->avatar_id;
        $image = '';
        if ($request->has('image')) {
            $saveFile = saveFile($request->image, HelperConstants::PROFILE_IMAGE_DIRECTORY, HelperConstants::UPLOAD_DISK);
            $image = $saveFile['fileName'];
            $user->image = $image;
        }

        $user->save();
        $title = $user->first_name . " " . $user->last_name . " just created a new account.";
        $body = $user->id;
        $data = [
            "user" => $user,
            "title" => $title,
            "body" => $user->id,
        ];
        $admin = User::where('type', 'admin')->first();
        $admin->notify(new SendAdminNotification($title, $body, $data));
        if ($request->has('images')) {
            $this->uploadVerificationImages($request, $user->id);
            $title = $user->first_name . " " . $user->last_name . " applied for varification.";
            $data = [
                "user" => $user,
                "title" => $title,
                "body" => $user->id,
            ];
            $admin->notify(new SendAdminNotification($title, $body, $data));
        }
        // Code to send notification to admin for approve this user
        return ['token' => $user->createToken('123', ['account:verified'])->plainTextToken, 'user' => new UserListingResource($user)];
        // $data = ['user' => new UserListingResource($user)];
        // return $data;
    }
    public function checkPassword($request): bool
    {
        return !Hash::check($request->password, auth()->user()->password);
    }

    public function updateFaceId($request): bool
    {
        auth()->user()->is_biometric_enable = 1;
        auth()->user()->face_id = $request->public_key;
        auth()->user()->save();
        return true;
    }

    public function uploadImages($request)
    {
        $user = User::find(auth()->user()->id);
        storeArrayOfImage($request, "images", User::IMAGES_DIRECTORY, $user, "images");
        return $user;
    }

    public function uploadSingleImage($request)
    {
        $user = User::find(auth()->user()->id);
        if ($request->hasFile('image')) {
            onlyStoreImage($request, 'image', User::IMAGES_DIRECTORY, $user, 'singleImage');
        }
        // storeArrayOfImage($request, "images", , $user, "images");
        return $user;
    }

    public function uploadVerificationImages($request, $user_id)
    {
        $user = User::find($user_id);
        storeArrayOfImage($request, "images", User::IMAGES_DIRECTORY, $user, "images", "verification");
        return $user;
    }

    public function tapProfile($id)
    {
        $loggedInUser = auth()->user();

        $message = "User added to tapped profiles!";
        $friendsIds = DB::table('friends')
            ->where('user_id', $loggedInUser->id)
            ->where('friend_id', $id)
            ->first();
        if ($friendsIds) {
            $message = "Profile already in chat";
            return $message;
        }
        $friendRequestsSentIds = DB::table('friend_requests')
            ->where('sender_id', $loggedInUser->id)
            ->where('receiver_id', $id)
            ->first();
        if ($friendRequestsSentIds) {
            $message = "Already message request sent";
            return $message;
        }
        $friendRequestsReceivedIds = DB::table('friend_requests')
            ->where('receiver_id', $loggedInUser->id)
            ->where('sender_id', $id)
            ->first();
        if ($friendRequestsReceivedIds) {
            $message = "Already recieved message request";
            return $message;
        }
        $user = $this->checkTappedProfile($id);
        if ($user) {
            $message = "Profile already added to tapped profiles";
            return $message;
        }
        $tapToMatch = new TapToMatch();
        $tapToMatch->tapped_user = $id;
        $tapToMatch->user_id = $loggedInUser->id;
        $tapToMatch->save();
        return $message;
    }

    public function removeTapProfile($id)
    {
        $message = "User removed from tapped profiles!";
        $loggedInUser = auth()->user();
        $user = $this->checkTappedProfile($id);
        if (!$user) {
            $message = "Profile does not exist in tapped profiles.";
            return $message;
        }
        $tapToMatch = TapToMatch::where('user_id', $loggedInUser->id)->where('tapped_user', $id)->first();
        $tapToMatch->delete();
        return $message;
    }

    // public function getTapProfiles($request)
    // {
    //     $tappedUsers = TapToMatch::where('user_id', auth()->user()->id)
    //         ->with([
    //             'tappedUser' => function ($query) use ($request) {
    //                 $query->when($request->has('search'), function ($query) use ($request) {
    //                     $search = $request->input('search');
    //                     // Apply search filter on multiple fields
    //                     $query->where(function ($q) use ($search) {
    //                         $q->where('first_name', 'like', '%' . $search . '%')
    //                             ->orWhere('last_name', 'like', '%' . $search . '%')
    //                             ->orWhere('email', 'like', '%' . $search . '%')
    //                             ->orWhere(DB::raw("CONCAT(first_name, ' ', last_name)"), 'like', '%' . $search . '%');
    //                     });
    //                 });
    //             }
    //         ])
    //         ->get()
    //         ->map(function ($tapToMatch) {
    //             return $tapToMatch->tappedUser; // Get the tappedUser from each TapToMatch record
    //         })
    //         ->filter(function ($tappedUser) {
    //             // Filter out any null values
    //             return $tappedUser !== null;
    //         })
    //         ->values(); // Reindex the collection after filtering
    //     $users = $tappedUsers->isEmpty() ? [] : $tappedUsers;
    //     // Return the collection, or an empty array if no results
    //     return $users;
    // }

    public function getTapProfiles($request)
    {
        $tappedUsers = TapToMatch::where('user_id', auth()->user()->id)
            ->with([
                'tappedUser' => function ($query) use ($request) {
                    // Search filter
                    $query->when($request->has('search'), function ($query) use ($request) {
                        $search = $request->input('search');
                        // Apply search filter on multiple fields
                        $query->where(function ($q) use ($search) {
                            $q->where('first_name', 'like', '%' . $search . '%')
                                ->orWhere('last_name', 'like', '%' . $search . '%')
                                ->orWhere('email', 'like', '%' . $search . '%')
                                ->orWhere(DB::raw("CONCAT(first_name, ' ', last_name)"), 'like', '%' . $search . '%');
                        });
                    });

                    // Age filter
                    if ($request->has('age_from') && $request->has('age_to')) {
                        $query->whereBetween('age', [$request->input('age_from'), $request->input('age_to')]);
                    }

                    // Gender filter
                    if ($request->has('gender')) {
                        $query->where('gender', $request->input('gender'));
                    }

                    // Ring filter
                    if ($request->has('ring')) {
                        $query->where('ring', $request->input('ring'));
                    }

                    // Preferences filter
                    if ($request->has('preferences')) {
                        $preferences = $request->input('preferences');  // Array of preferences (question_id + answer)
        
                        foreach ($preferences as $preference) {
                            // Each preference should have a 'question_id' and 'answer' (array of selected question_option_ids)
                            if (isset($preference['question_id']) && isset($preference['answer'])) {
                                if (is_array($preference['answer'])) {
                                    $preference['answer'] = array_map('intval', $preference['answer']);
                                }
                                // $preference['answer'] = array_map('intval', $preference['answer']); // Convert all values in the answer array to integers
        
                                $query->whereHas('preferences', function ($q) use ($preference) {
                                    $q->where('question_id', $preference['question_id'])
                                        ->whereJsonContains('answer', $preference['answer']);  // Expecting $preference['answer'] to be an array of integers
                                });
                            }
                        }
                    }
                    if ($request->has('height_from') && $request->has('height_to')) {
                        // Convert the height range inputs from "4'1\"" to inches
                        $heightFrom = $this->convertHeightToInches($request->input('height_from'));  // E.g., 49 inches
                        $heightTo = $this->convertHeightToInches($request->input('height_to'));      // E.g., 65 inches
        
                        // Log the converted heights for debugging purposes
                        \Log::debug("Height From (in inches): $heightFrom, Height To (in inches): $heightTo");

                        // Query the database to get users with height in the given range (in inches)
                        $query->whereRaw("(
        CAST(SUBSTRING_INDEX(SUBSTRING_INDEX(height, '\'', 1), '\'', -1) AS UNSIGNED) * 12 +
        CAST(SUBSTRING_INDEX(height, '\'', -1) AS UNSIGNED)
    ) BETWEEN ? AND ?", [$heightFrom, $heightTo]);
                    }

                }
            ])
            ->get()
            ->map(function ($tapToMatch) {
                return $tapToMatch->tappedUser; // Get the tappedUser from each TapToMatch record
            })
            ->filter(function ($tappedUser) {
                // Filter out any null values
                return $tappedUser !== null;
            })
            ->values(); // Reindex the collection after filtering

        $users = $tappedUsers->isEmpty() ? [] : $tappedUsers;

        // Return the collection, or an empty array if no results
        return $users;
    }
    private function convertHeightToInches($height)
    {
        // Match the format "4'1\"" where one single quote and one double quote exist
        if (preg_match("/(\d+)\'(\d+)\"/", $height, $matches)) {
            $feet = (int) $matches[1];  // Feet part (before the single quote)
            $inches = (int) $matches[2];  // Inches part (after the single quote and before the double quote)

            // Convert to inches: (feet * 12) + inches
            return ($feet * 12) + $inches;
        }

        return 0;  // Return 0 if the format is invalid
    }
    public function getTapProfilesCount()
    {
        $tappedUsers = TapToMatch::where('user_id', auth()->user()->id)->count();
        return $tappedUsers;
    }

    public function checkTappedProfile($id)
    {
        $user = TapToMatch::where('tapped_user', $id)->where('user_id', auth()->user()->id)->first();
        return $user;
    }


    public function deleteImage($image)
    {
        deleteImage($image);
        return auth()->user();
    }


    public function uploadSocialLinks($request)
    {
        $user = auth()->user();

        // Separate data for updating and inserting
        $updateData = [];
        $insertData = [];
        $typesInRequest = [];

        // Collect types from the request to filter existing links
        foreach ($request->social_links as $socialLink) {
            $typesInRequest[] = $socialLink['type'];
        }

        // Get existing social links for the types provided in the request
        $existingLinks = $user->socialLinks()->whereIn('type', $typesInRequest)->get()->keyBy('type');

        foreach ($request->social_links as $socialLink) {
            if ($existingLinks->has($socialLink['type'])) {
                // Prepare data for updating
                $updateData[] = [
                    'id' => $existingLinks[$socialLink['type']]->id,
                    'link' => $socialLink['link']
                ];
            } else {
                // Prepare data for inserting
                $insertData[] = [
                    'link' => $socialLink['link'],
                    'type' => $socialLink['type'],
                    'user_id' => $user->id,
                    'created_at' => now(),
                    'updated_at' => now()
                ];
            }
        }

        // Perform batch update for existing links
        if (!empty($updateData)) {
            foreach ($updateData as $data) {
                \DB::table('user_social_links')
                    ->where('id', $data['id'])
                    ->update(['link' => $data['link'], 'updated_at' => now()]);
            }
        }

        // Perform batch insert for new links
        if (!empty($insertData)) {
            \DB::table('user_social_links')->insert($insertData);
        }
        return $user;
    }
    public function disableFaceId(): bool
    {
        auth()->user()->is_biometric_enable = 0;
        auth()->user()->face_id = "";
        auth()->user()->save();
        return true;
    }

    public function enable_location($userId)
    {
        $user = User::find($userId);
        $user->is_location_enable = ($user->is_location_enable == 1) ? 0 : 1;
        $user->save();
        return $user;
    }

    public function changeStatusOfAProperty($userId, $property)
    {
        $user = User::find($userId);
        $user->$property = ($user->$property == 1) ? 0 : 1;
        $user->save();
        return $user;
    }

    public function updateNfc($userId, $request)
    {
        $user = User::find($userId);
        $user->nfc_tag = $request->nfc_tag;
        $user->save();
        return "https://bandeddating.com/view-profile-nfc-details/$user->id";
    }

    public function updateLocation($userId, $request)
    {
        $user = User::find($userId);
        $user->longitude = $request->longitude;
        $user->latitude = $request->latitude;
        $user->save();
        return $user;
    }

    public function updatePinpointLocation($userId, $longitude, $latitude)
    {
        $user = User::find($userId);
        $user->pintpont_longitude = $longitude;
        $user->pintpont_latitude = $latitude;
        $user->save();
        return $user;
    }

    public function update($request)
    {
        auth()->user()->name = $request->name;
        auth()->user()->dialing_code = $request->dialing_code;
        auth()->user()->phone = $request->phone;
        auth()->user()->dob = $request->dob;
        auth()->user()->country = $request->country;

        if ($request->has('image')) {
            $saveFile = saveFile($request->image, HelperConstants::PROFILE_image_DIRECTORY, HelperConstants::UPLOAD_DISK);
            auth()->user()->image = $saveFile['fileName'];
        }

        auth()->user()->save();

        return auth()->user();
    }

    public function getAll()
    {
        return User::all();
    }

    public function show($id)
    {
        return User::find($id);
    }

    public function updatePassword($request, $id)
    {
        $user = User::find($id);

        if ($request->has('password')) {
            $user->password = Hash::make($request->password);
        }
        $user->save();

        return $user;
    }

    public function updateStatus($request, $id)
    {
        $user = User::find($id);
        $user->status = $request->status;
        $user->save();

        return $user;
    }

    public function destroy($id)
    {
        DB::beginTransaction();
        User::find($id)->delete();
        DB::commit();
        return response()->json(['status' => true, 'message' => 'deleted successfully']);
    }

    public function getUserByEmail($email)
    {
        return User::where('email', $email)->first();
    }

    public function getUserByPhone($phone)
    {
        return User::where('phone', $phone)->first();
    }

    public function getUserByFaceId($face_id)
    {
        return User::where('face_id', $face_id)->first();
    }

    public function checkLogin($request, $type = 'user')
    {

        if ($request->exists('face_id') && $request->filled('phone')) {
            $user = $this->getUserByFaceId($request->face_id);

            if (!$user) {
                throw ValidationException::withMessages([
                    'phone' => ['Incorrect face Id'],
                ]);
            }
        } else {
            $user = $this->getUserByEmail($request->email);
            if (!$user || !Hash::check($request->password, $user->password)) {
                throw new AuthenticationException(
                    "The provided credentials are incorrect."
                );
            }
        }
        if ($user->type !== $type) {
            throw new AuthenticationException(
                "You're not valid user to access this route"
            );
        }
        if ($user->email_verified_at == null) {
            throw new AuthenticationException(
                "Your account is not verified"
            );
        }
        if ($user->status == 0) {
            throw new AuthenticationException(
                "Your account has been deactivated"
            );
        }
        if ($request->latitude && $request->longitude) {
            $this->updateDeviceLocationByEmail($user->email, $request->latitude, $request->longitude);
        }
        $user = User::with(['images', 'verificationimages', 'socialLinks', 'userAnswers', 'userAnswers.question.category'])->find($user->id);

        $this->updateDeviceToken($user->email, $request->device_token);
        return ['token' => $user->createToken('123', ['account:verified'])->plainTextToken, 'user' => new UserProfileWithAllDataResource($user)];
    }

    public function checkBiometricLogin($request, $type = 'user')
    {
        $email = $request->email;
        $payload = $request->input('payload');
        $signature = $request->input('signature');

        // Fetch the public key associated with the external identifier
        $publicKey = DB::table('users')->where('email', $email)->value('face_id');

        if (!$publicKey) {
            $this->removePublicKey($request->email);
            throw new AuthenticationException(
                "Public key not found"
            );
        }
        $formattedPublicKey = "-----BEGIN PUBLIC KEY-----\n" . wordwrap($publicKey, 64, "\n", true) . "\n-----END PUBLIC KEY-----";
        $isValid = openssl_verify($payload, base64_decode($signature), $formattedPublicKey, OPENSSL_ALGO_SHA256);

        if ($isValid) {
            $user = $this->getUserByEmail($request->email);
            if ($user->type !== $type) {
                throw new AuthenticationException(
                    "You're not valid user to access this route"
                );
            }
            if ($user->email_verified_at == null) {
                throw new AuthenticationException(
                    "Your account is not verified"
                );
            }
            if ($user->status == 0) {
                throw new AuthenticationException(
                    "Your account has been deactivated"
                );
            }
            $this->updateDeviceToken($user->email, $request->device_token);
            if ($request->latitude && $request->longitude) {
                $this->updateDeviceLocationByEmail($user->email, $request->latitude, $request->longitude);
            }
            return ['token' => $user->createToken('123', ['account:verified'])->plainTextToken, 'user' => new UserListingResource($user)];
        } else {
            $this->removePublicKey($request->email);
            throw new AuthenticationException(
                "Invalid payload"
            );
        }
    }
    public function removePublicKey($email)
    {
        $user = User::where('email', $email)->first();
        if ($user) {
            $user->is_biometric_enable = 0;
            $user->face_id = '';
            $user->save();
        }
    }
    public function updateDeviceToken($email, $deviceToken)
    {
        $user = $this->getUserByEmail($email);
        $user->device_token = $deviceToken;
        $user->save();
    }

    public function updateDeviceTokenByPhone($phone, $deviceToken)
    {
        $user = $this->getUserByPhone($phone);
        $user->device_token = $deviceToken;
        $user->save();
    }


    public function updateDeviceLocationByEmail($email, $latitude, $longitude)
    {
        $user = $this->getUserByEmail($email);
        $user->longitude = $longitude;
        $user->latitude = $latitude;
        $user->save();
    }

    public function reset($request)
    {
        $user = $this->getUserByEmail($request->email);

        if (!$user) {
            throw ValidationException::withMessages([
                'email' => ['Email is not registered'],
            ]);
        }

        $passwordReset = $this->generateToken($user->email);

        if ($user && $passwordReset)
            $user->notify(
                new PasswordResetRequest($user, $passwordReset->token)
            );
        return true;

    }

    public function verifyAndUpdatePassword($request)
    {
        $user = $this->getUserByEmail($request->email);
        return $this->updatePassword($request, $user->id);
    }

    public function sendAccountVerification($user)
    {
        $verification = $this->generateToken($user->email);
        if ($user && $verification)
            $user->notify(
                new SignupActivation($user, $verification->token)
            );
    }

    public function generateToken($email)
    {
        return PasswordReset::updateOrCreate(
            ['email' => $email],
            [
                'email' => $email,
                'token' => strtolower(rand(1000, 9999)),
                'created_at' => now()
            ]
        );
    }

    public function markVerify($request, $email)
    {
        $code = $this->getToken($request);

        if (!$code || $code->email != $email) {
            return false;
        }
        $user = $this->getUserByEmail($email);

        $user->verified = 1;

        // Remove after development of admin
        $user->status = 1;

        $user->save();

        $this->deleteToken($request->token);

        return true;
    }

    public function getToken($request)
    {
        return PasswordReset::where('token', $request->token)->first();
    }

    public function deleteToken($token)
    {
        PasswordReset::where('token', $token)->delete();
    }

    public function setLocation($request, $userId)
    {
        $user = User::where('id', $userId)->first();

        if ($user->location) {

            $location['latitude'] = $request->latitude;
            $location['longitude'] = $request->longitude;
            $location['text'] = $request->location;
            $user->location()->update($location);
        } else {

            $location = new \App\Models\Location;
            $location->latitude = $request->latitude;
            $location->longitude = $request->longitude;
            $location->text = $request->location;
            $user->location()->save($location);
        }

        return $user;
    }

    public function pushNotificationStatus($userId, $status)
    {
        $user = User::find($userId);
        $user->push_notification = $status;
        $user->save();

        return $user;
    }

    public function logout($request)
    {
        return $request->user()->currentAccessToken()->delete();
    }

    public function listing($request, $pagination)
    {
        return User
            ->whereNotIn('status', [User::PENDING_STATUS, User::REJECT_STATUS])
            ->when(request()->filled('status'), function ($q) {
                $q->whereStatus(request('status'));
            })
            ->when(request()->filled('search'), function ($q) {
                $q->where('name', 'like', '%' . request("search") . '%');
            })
            ->when(request()->filled('start'), function ($q) {
                $q->whereDate('created_at', '>=', request('start'))->whereDate('created_at', '<=', request('end'));
            })
            ->orderBy('id', 'DESC')
            ->paginate($pagination);
    }

    public function requestListing($request, $pagination)
    {
        return User::
            where(function ($q) {
                $q->where('status', User::PENDING_STATUS)->orWhere('status', User::REJECT_STATUS);
            })
            ->when(request()->filled('status'), function ($q) {
                $q->whereStatus(request('status'));
            })
            ->when(request()->filled('search'), function ($q) {
                $q->where('name', 'like', '%' . request("search") . '%');
            })
            ->when(request()->filled('start'), function ($q) {
                $q->whereDate('created_at', '>=', request('start'))->whereDate('created_at', '<=', request('end'));
            })
            ->orderBy('id', 'DESC')
            ->paginate($pagination);
    }

    public function accountStatus($userId, $status)
    {
        $user = User::find($userId);
        $user->status = $status;
        $user->save();

        return $user;
    }

    public function updateUserLocationAndRadius($userId, $latitude, $longitude, $radius = 10)
    {
        $location = UserLocation::updateOrCreate(
            ['user_id' => $userId],
            ['latitude' => $latitude, 'longitude' => $longitude, 'radius' => $radius]
        );

        return $location;
    }
}
