<?php

namespace App\Services;

use App\Exceptions\BadRequestException;
use App\Models\CancelledUser;
use App\Models\Friend;
use App\Models\FriendRequest;
use App\Models\TapToMatch;
use App\Models\User;
use App\Services\Common\BaseService;
use Carbon\Carbon;
use Exception;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\DB;

/**
 * Class JsonResponseService
 * @package App\Services
 */
class FriendService extends BaseService
{

    public function sendRequest($receiverId)
    {
        DB::beginTransaction();
        $senderId = auth()->user()->id;
        if ($senderId == $receiverId) {
            throw new BadRequestException('You cannot send a message request to yourself.');
        }

        // Check if the user has already sent 5 friend requests today
        $requestCountToday = FriendRequest::where('sender_id', $senderId)
            ->whereDate('created_at', now()->toDateString())
            ->count();

        if ($requestCountToday >= 5) {
            throw new BadRequestException('You have reached your limit of message requests for today.');
        }

        // Check if there's already an accepted friend request between these users
        $existingRequest = FriendRequest::where('sender_id', $senderId)
            ->where('receiver_id', $receiverId)
            ->where('status', 'accepted')
            ->first();

        if ($existingRequest) {
            throw new BadRequestException('Message request already sent.');
        }

        // Create a new friend request
        FriendRequest::create([
            'sender_id' => $senderId,
            'receiver_id' => $receiverId,
            'status' => 'pending',
        ]);
        $tapToMatchUser = TapToMatch::where('tapped_user', $receiverId)->where('user_id', $senderId)->first();
        if ($tapToMatchUser) {
            $tapToMatchUser->delete();
        }

        // $title = $user->first_name . " " . $user->last_name . " created order !";
        // $body = $user->id;
        // $data = [
        //     "user_id" => $user->id,
        //     "title" => $user->first_name . " " . $user->last_name . " created order !",
        //     "body" => $user->id,
        // ];

        // sendNotificationToAdmin($title, $body, $data);
        DB::commit();
        return 'Message request sent.';
    }

    public function addToCancelledUsers($receiverId)
    {
        $cancelledUser = new CancelledUser();
        $cancelledUser->user_id = auth()->user()->id;
        $cancelledUser->cancelled_user = $receiverId;
        $cancelledUser->save();
        return $cancelledUser;
    }

    public function updateCancelledUserCreatedAtTime($id)
    {
        $cancelledUser = CancelledUser::where('id', $id)->first();
        $cancelledUser->created_at = Carbon::now();
        $cancelledUser->save();
        return $cancelledUser;
    }

    public function checkIfAlreadyInCancelledRequests($receiverId)
    {
        return CancelledUser::where('user_id', auth()->user()->id)->where('cancelled_user', $receiverId)->first();
    }

    public function getDailyFriendRequestCount()
    {
        $userId = auth()->user()->id;

        // Count friend requests sent by the user on the current day
        $dailyRequestCount = FriendRequest::where('sender_id', $userId)
            ->whereDate('created_at', now()->toDateString())
            ->count();

        return $dailyRequestCount;
    }

    public function acceptRequest($requestId, $chatService)
    {
        $request = FriendRequest::find($requestId);

        if (!$request || $request->receiver_id != auth()->user()->id) {
            throw ValidationException::withMessages([
                'Request not found or unauthorized.',
            ]);
        }

        $request->update(['status' => 'accepted']);

        // Add to friends table
        Friend::create([
            'user_id' => $request->sender_id,
            'friend_id' => $request->receiver_id,
        ]);
        Friend::create([
            'user_id' => $request->receiver_id,
            'friend_id' => $request->sender_id,
        ]);
        $chatService->createChatWithRequestAccept($request->sender_id, $request->receiver_id);
        return 'Message request accepted.';
    }

    public function rejectRequest($requestId)
    {

        $request = FriendRequest::find($requestId);
        if (!$request || $request->receiver_id != auth()->user()->id) {
            throw ValidationException::withMessages([
                'Request not found or unauthorized.',
            ]);
        }

        $request->delete();

        return 'Message request rejected.';
    }

    public function cancelRequest($userId)
    {
        $request = FriendRequest::where('sender_id', auth()->user()->id)->where('receiver_id', $userId)->first();

        if (!$request) {
            throw ValidationException::withMessages([
                'Request not found or unauthorized.',
            ]);
        }

        $request->update(['status' => 'cancelled']);

        return 'Message request cancelled.';
    }

    public function getFriendsList()
    {
        $userId = auth()->user()->id;

        // Fetch friends of the authenticated user
        $friends = User::whereIn('id', function ($query) use ($userId) {
            $query->select('friend_id')
                ->from('friends')
                ->where('user_id', $userId);
        })
            // Exclude users who are in the blocked_user table
            // ->whereNotIn('id', function ($query) use ($userId) {
            //     $query->select('blocked_user_id')
            //         ->from('blocked_users')
            //         ->where('user_id', $userId);
            // })
            ->paginate($this->pagination);

        return $friends;
    }

    public function friendRequests()
    {
        $userId = auth()->user()->id;
        // Fetch friends of the authenticated user
        $search = request("search");
        $friends = FriendRequest::where('receiver_id', $userId)
            ->where('status', 'pending')
            ->with('sender')
            ->when(request()->filled('search'), function ($query) use ($search) {
                $query->whereHas('sender', function ($q) use ($search) {
                    $q->where('first_name', 'like', '%' . request("search") . '%')
                        ->orWhere('last_name', 'like', '%' . request("search") . '%')
                        ->orWhere('email', 'like', '%' . request("search") . '%')
                        ->orWhere(DB::raw("CONCAT(first_name, ' ', last_name)"), 'like', '%' . request("search") . '%');
                });
            })
            ->paginate($this->pagination);
        return $friends;
    }

    public function friendRequestsCount()
    {
        $userId = auth()->user()->id;
        // Fetch friends of the authenticated user
        $friends = FriendRequest::where('receiver_id', $userId)
            ->where('status', 'pending')
            ->count();
        return $friends;
    }



    public function unFriend($id)
    {
        DB::beginTransaction();
        // $userId = auth()->user()->id;
        // // Fetch friends of the authenticated user
        // $friendRequest = FriendRequest::where('receiver_id', $userId)->where('sender_id', $id)->where('status', 'accepted')->first();
        // if (!$friendRequest) {
        //     $friendRequest = FriendRequest::where('sender_id', $userId)->where('receiver_id', $id)->where('status', 'accepted')->first();
        // }
        // $chatService = new ChatService();

        // $friend1 = Friend::where('user_id', $userId)->where('friend_id', $id)->first();
        // $friend2 = Friend::where('friend_id', $userId)->where('user_id', $id)->first();
        // $response = $chatService->deleteChat($userId, $id);
        // $friendRequest->delete();
        // dd($friend1->delete());
        // $friend2->delete();

        // DB::commit();
        // return "Friend delete.";

        try {
            $userId = auth()->user()->id;

            // Fetch the accepted friend request
            $friendRequest = FriendRequest::where('receiver_id', $userId)
                ->where('sender_id', $id)
                ->where('status', 'accepted')
                ->first();

            if (!$friendRequest) {
                $friendRequest = FriendRequest::where('sender_id', $userId)
                    ->where('receiver_id', $id)
                    ->where('status', 'accepted')
                    ->first();
            }

            if (!$friendRequest) {
                throw new Exception('Message request not found.');
            }

            $chatService = new ChatService();

            // Fetch the friendships
            $friend1 = Friend::where('user_id', $userId)->where('friend_id', $id)->first();
            $friend2 = Friend::where('friend_id', $userId)->where('user_id', $id)->first();
            // Ensure the friend records exist before attempting to delete
            if ($friend1) {
                $friend1->delete();
            }

            if ($friend2) {
                $friend2->delete();
            }

            // Delete the friend request
            $friendRequest->delete();

            // Delete the chat (ensure this method is correct and returns a valid response)
            $response = $chatService->deleteChat($userId, $id);

            // Commit the transaction


            return "Friend deleted successfully.";
        } catch (Exception $e) {
            // Rollback in case of any error
            DB::rollBack();

            // Log the error message or handle it as needed
            return "An error occurred: " . $e->getMessage();
        }
    }
}
