<?php

namespace App\Services;

use App\Models\Cart;
use App\Models\DeliveryCharges;
use App\Models\Order;
use App\Models\OrderAddress;
use App\Models\OrderDetail;
use App\Models\Store;
use App\Models\StoreCategory;
use App\Models\User;
use App\Services\Common\BaseService;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Facades\DB;
use Mockery\Exception;
use Symfony\Component\HttpFoundation\Response;

/**
 * Class JsonResponseService
 * @package App\Services
 */
class OrderService extends BaseService
{

    public function index()
    {

        $orders = Order::with([
            'user',
            'details' =>
                function ($q) {
                    $q->selectRaw('*, (price * quantity) as total_price');
                }
        ])->whereIn('status', [Order::PENDING, Order::DELIVERED, Order::CANCELLED])
            ->when(request()->filled('search'), function ($q) {
                $search = request('search');

                $q->where(function ($q) use ($search) {
                    $q->where('id', 'like', '%' . $search . '%')
                        ->orWhereHas('user', function ($query) use ($search) {
                            $query->where('first_name', 'like', '%' . $search . '%')
                                ->orWhere('last_name', 'like', '%' . $search . '%')
                                ->orWhere('email', 'like', '%' . $search . '%')
                                ->orWhere(DB::raw("CONCAT(first_name, ' ', last_name)"), 'like', '%' . $search . '%');
                        });
                });
            })
            ->when(request()->filled('status') && request('status') != 0, function ($q) {
                $q->where('status', request("status"));
            })
            ->when(request()->filled('fromDate'), function ($q) {
                $q->whereDate('created_at', '>=', request('fromDate'))
                    ->whereDate('created_at', '<=', request('toDate'));
            })
            ->orderBy('id', 'desc')
            ->paginate($this->pagination);
        return $orders;
    }


    public function userOrders($id)
    {
        $orders = Order::with([
            'user',
            'details' =>
                function ($q) {
                    $q->selectRaw('*, (price * quantity) as total_price');
                }
        ])->where('user_id', $id)
            ->whereIn('status', [Order::PENDING, Order::DELIVERED, Order::CANCELLED])
            ->when(request()->filled('search'), function ($q) {
                $search = request('search');
                $q->where(function ($q) use ($search) {
                    $q->where('id', 'like', '%' . $search . '%')
                        ->orWhereHas('user', function ($query) use ($search) {
                            $query->where('first_name', 'like', '%' . $search . '%')
                                ->orWhere('last_name', 'like', '%' . $search . '%')
                                ->orWhere('email', 'like', '%' . $search . '%')
                                ->orWhere(DB::raw("CONCAT(first_name, ' ', last_name)"), 'like', '%' . $search . '%');
                        });
                });
            })
            ->when(request()->filled('status') && request('status') != 0, function ($q) {
                $q->where('status', request("status"));
            })
            ->when(request()->filled('fromDate'), function ($q) {
                $q->whereDate('created_at', '>=', request('fromDate'))
                    ->whereDate('created_at', '<=', request('toDate'));
            })
            ->orderBy('id', 'desc')
            ->paginate($this->pagination);
        return $orders;
    }


    public function show($id)
    {
        $order = Order::where('id', $id)->with([
            'details' =>
                function ($q) {
                    $q->selectRaw('*, (price * quantity) as total_price');
                }
        ], 'address')->first();
        return $order;
    }

    public function changeStatus($request)
    {
        $order = Order::where('id', $request->order_id)->first();
        $user = User::find($order->user_id);
        $order->status = $request->status;
        if ($request->status === "cancelled") {
            $order->rejection_reason = $request->reason;
        }
        $order->save();
        $title = $user->first_name . " " . $user->last_name . " your order has been $order->status !";
        $body = $user->id;
        $data = [
            "user" => $user,
            "title" => $title,
            "body" => $user->id,
        ];
        sendNotificationToUser($title, $body, $data, $user);
        return $order;
    }

    public function store($request, $stripeService)
    {
        DB::beginTransaction();
        $user_id = request('user_id');
        $deliveryCharge = DeliveryCharges::where('is_default', 1)->first();
        $orderDetails = array();
        $address = array();
        $cart = Cart::selectRaw('*, (price * quantity) as total_price')
            ->where('user_id', $user_id)
            ->get();
        $totalAmount = collect($cart)->sum('total_price') + $deliveryCharge->amount;
        $order = new Order();
        $order->total_price = $totalAmount;
        $order->total_quantity = $cart->count();
        $order->status = Order::REJECTED;
        $order->rejection_reason = "Payment not recieved";
        $order->user_id = $user_id;
        $order->save();

        foreach ($cart as $key => $value) {
            $orderDetails[$key]['price'] = $value['price'];
            $orderDetails[$key]['quantity'] = $value['quantity'];
            $orderDetails[$key]['product_id'] = $value['product_id'];
            $orderDetails[$key]['size_id'] = $value['size_id'];
            $orderDetails[$key]['order_id'] = $order->id;
        }
        OrderDetail::insert($orderDetails);
        //Order shipping Address
        $shippingAddress = $request->shippingAddress;
        $shippingAddress['email'] = '';
        $shippingAddress['type'] = OrderAddress::SHIPPING_ADDRESS; // 'shipping'
        $shippingAddress['order_id'] = $order->id;
        $address[0] = $shippingAddress;

        // Order Billing Address
        $billingAddress = $request->billingAddress;
        $billingAddress['email'] = '';
        $billingAddress['type'] = OrderAddress::BILLING_ADDRESS; // 'billing'
        $billingAddress['order_id'] = $order->id;
        $address[1] = $billingAddress;

        // Order Contact Information
        $contactInfo = $request->contactInfo;

        // Ensure required fields are set correctly, if your logic allows
        $contactInfo['address'] = ''; // If you don't need this, ensure your database allows NULL
        $contactInfo['country'] = ''; // Ensure it's set or handled according to your logic
        $contactInfo['city'] = ''; // Ensure it's set or handled according to your logic
        $contactInfo['state'] = ''; // Ensure it's set or handled according to your logic
        $contactInfo['zip_code'] = ''; // Ensure it's set or handled according to your logic

        // Set the contact info type and order ID
        $contactInfo['type'] = OrderAddress::CONTACT_INFORMATION; // 'contact'
        $contactInfo['order_id'] = $order->id;

        // Construct the address array
        $address[2] = $contactInfo;
        // Insert addresses into the database
        OrderAddress::insert($address);
        $stripeCheckoutSession = $stripeService->createCheckoutSession($totalAmount, $user_id, $order->id, $orderDetails, $deliveryCharge->amount);
        Cart::where('user_id', $user_id)->delete();
        DB::commit();
        return $stripeCheckoutSession;
    }
}
