<?php

namespace App\Services;

use App\Models\Product;
use App\Models\ProductSize;
use App\Models\Store;
use App\Models\StoreCategory;
use App\Models\User;
use App\Services\Common\BaseService;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Facades\DB;
use Mockery\Exception;
use Symfony\Component\HttpFoundation\Response;

/**
 * Class JsonResponseService
 * @package App\Services
 */
class ProductService extends BaseService
{

    public function index()
    {
        $products = Product::where('status', 'active')->when(request()->filled('search'), function ($q) {
            $q->where(function ($q) {
                $q->where('name', 'like', '%' . request("search") . '%');
            });
        })
            ->when(request()->filled('status'), function ($q) {
                $q->where(function ($q) {
                    $q->where('status', request("status"));
                });
            })
            ->when(request()->filled('categories'), function ($q) {
                $q->where(function ($q) {
                    $q->whereIn('category_id', request("categories"));
                });
            })
            ->when(request()->filled('fromDate'), function ($q) {
                $q->whereDate('created_at', '>=', request('fromDate'))->whereDate('created_at', '<=', request('toDate'));
            })
            ->orderBy('id', 'desc')
            ->paginate($this->pagination);
        return $products;
    }

    public function byCategory($id)
    {
        $products = Product::where('status', 'active')->where('category_id', $id)->when(request()->filled('search'), function ($q) {
            $q->where(function ($q) {
                $q->where('name', 'like', '%' . request("search") . '%');
            });
        })
            ->when(request()->filled('status'), function ($q) {
                $q->where(function ($q) {
                    $q->where('status', request("status"));
                });
            })
            ->when(request()->filled('categories'), function ($q) {
                $q->where(function ($q) {
                    $q->whereIn('category_id', request("categories"));
                });
            })
            ->when(request()->filled('fromDate'), function ($q) {
                $q->whereDate('created_at', '>=', request('fromDate'))->whereDate('created_at', '<=', request('toDate'));
            })
            ->orderBy('id', 'desc')
            ->paginate($this->pagination);
        return $products;
    }

    public function adminProducts()
    {
        $products = Product::when(request()->filled('search'), function ($q) {
            $q->where(function ($q) {
                $q->where('name', 'like', '%' . request("search") . '%');
            });
        })
            ->when(request()->filled('status'), function ($q) {
                $q->where(function ($q) {
                    $q->where('status', request("status"));
                });
            })
            ->when(request()->filled('Category'), function ($q) {
                $q->where(function ($q) {
                    $q->where('category_id', request("Category"));
                });
            })
            ->when(request()->filled('fromDate'), function ($q) {
                $q->whereDate('created_at', '>=', request('fromDate'))->whereDate('created_at', '<=', request('toDate'));
            })
            ->orderBy('id', 'desc')
            ->paginate($this->pagination);
        return $products;
    }

    public function userProducts()
    {
        $products = Product::when(request()->filled('search'), function ($q) {
            $q->where(function ($q) {
                $q->where('name', 'like', '%' . request("search") . '%');
            });
        })
            ->when(request()->filled('status'), function ($q) {
                $q->where(function ($q) {
                    $q->where('status', request("status"));
                });
            })
            ->when(request()->filled('categories'), function ($q) {
                $q->where(function ($q) {
                    $q->whereIn('category_id', request("categories"));
                });
            })
            ->when(request()->filled('fromDate'), function ($q) {
                $q->whereDate('created_at', '>=', request('fromDate'))->whereDate('created_at', '<=', request('toDate'));
            })
            ->orderBy('id', 'desc')
            ->paginate($this->pagination);
        return $products;
    }


    public function activeProducts()
    {
        $products = Product::where('status', 'active')
            ->with(relations: 'isInWishlist')
            ->when(request()->filled('search'), function ($q) {
                $q->where(function ($q) {
                    $q->where('name', 'like', '%' . request("search") . '%');
                });
            })
            ->when(request()->filled('status'), function ($q) {
                $q->where(function ($q) {
                    $q->where('status', request("status"));
                });
            })
            ->when(request()->filled('fromDate'), function ($q) {
                $q->whereDate('created_at', '>=', request('fromDate'))->whereDate('created_at', '<=', request('toDate'));
            })
            ->orderBy('id', 'desc')
            ->paginate($this->pagination);
        return $products;
    }

    public function categoryProducts($category_id)
    {
        $products = Product::where('category_id', $category_id)
            ->when(request()->filled('search'), function ($q) {
                $q->where(function ($q) {
                    $q->where('name', 'like', '%' . request('search') . '%');
                });
            })
            ->when(request()->filled('status'), function ($q) {
                $q->where(function ($q) {
                    $q->where('status', request("status"));
                });
            })
            ->when(request()->filled('fromDate'), function ($q) {
                $q->whereDate('created_at', '>=', request('fromDate'))
                    ->whereDate('created_at', '<=', request('toDate'));
            })
            ->orderBy('id', 'desc')
            ->paginate($this->pagination);

        return $products;
    }


    public function show($id)
    {
        $product = Product::where('id', $id)->first();
        return $product;
    }

    public function deleteProductImage($image)
    {
        deleteImage($image);
        $product = Product::where('id', $image->imageable_id)->with('images')->first();
        return $product->images;
    }

    public function store($request)
    {
        DB::beginTransaction();
        $product = new Product();
        $product->name = $request->name;
        $product->price = $request->price;
        $product->discounted_price = $request->discounted_price;
        $product->quantity = $request->quantity;
        $product->sku = $request->sku;
        $product->status = $request->status;
        $product->description = $request->description;
        $product->category_id = $request->category_id;

        // if ($request->has('images')) {
        //     $saveFile = saveFile($request->image, \HelperConstants::PRODUCT_IMAGE_DIRECTORY, \HelperConstants::UPLOAD_DISK);
        //     $image = $saveFile['fileName'];
        //     $product->image = $image;
        // }
        $product->save();
        if ($request->hasFile('images')) {
            storeArrayOfImage($request, "images", Product::IMAGES_DIRECTORY, $product, "images");
        }
        $productSizes = [];
        foreach ($request->size as $size) {
            $productSizes[] = [
                'product_id' => $product->id,
                'size_id' => $size['value'],
            ];
        }
        DB::table('product_sizes')->insert($productSizes);
        DB::commit();
        return $product;
    }

    public function update($request, $product)
    {
        DB::beginTransaction();
        $product->name = $request->name;
        $product->price = $request->price;
        $product->discounted_price = $request->discounted_price;
        $product->quantity = $request->quantity;
        $product->sku = $request->sku;
        $product->status = $request->status;
        $product->description = $request->description;
        $product->category_id = $request->category_id;
        if ($request->has('image') && $request->image != null) {
            $saveFile = saveFile($request->image, \HelperConstants::PRODUCT_IMAGE_DIRECTORY, \HelperConstants::UPLOAD_DISK);
            $image = $saveFile['fileName'];
            $product->image = $image;
        }
        $product->save();
        ProductSize::where('product_id', $product->id)->delete();
        $productSizes = [];
        foreach ($request->size as $size) {
            $productSizes[] = [
                'product_id' => $product->id,
                'size_id' => $size['value'],
            ];
        }
        DB::table('product_sizes')->insert($productSizes);
        if ($request->hasFile('images')) {
            storeArrayOfImage($request, "images", Product::IMAGES_DIRECTORY, $product, "images");
        }
        DB::commit();
        return $product;
    }


    public function ChangeStatus($id)
    {
        $product = Product::find($id);
        $product->status = ($product->status == Product::ACTIVE) ? Product::INACTIVE : Product::ACTIVE;
        $product->save();
        return $product;
    }
}
