<?php


namespace App\Services;

use App\Models\DeliveryCharges;
use App\Models\Order;
use App\Models\OrderAddress;
use App\Models\OrderDetail;
use App\Models\PaymentLog;
use App\Models\Product;
use App\Models\Shipment;
use App\Models\User;
use App\Notifications\OrderNotification;
use App\Notifications\PushNotification;
use App\Notifications\SendAdminNotification;
use App\Notifications\SendConfirmShipmentNotificationToGuestUser;
use Illuminate\Support\Facades\Notification;
use Stripe\Stripe;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class StripeService
{
    protected $stripe;

    public function __construct()
    {
        $this->stripe = new \Stripe\StripeClient(config('stripe.secret_key'));
    }

    public function createCustomer($name, $email)
    {
        return $this->stripe->customers->create([
            'name' => $name,
            'email' => $email,
        ]);
    }

    public function createCheckoutSession($amount, $user_id, $order_id, $orderDetails, $deliveryCharge)
    {
        foreach ($orderDetails as $item) {
            $line_items[] = [
                'quantity' => $item['quantity'],
                'price_data' => [
                    'unit_amount' => (((int) ($item['price'] * 100))),
                    'currency' => 'usd',
                    'product_data' => [
                        'name' => "Product Payment",
                        'images' => ['https://files.stripe.com/links/MDB8YWNjdF8xS2Vld2tLY2lOUHR6b1RpfGZsX3Rlc3RfYkZzRkM5UkxYMTFRUHFTQ21mNWM2ZFQx00GF3YS6op'],
                    ]
                ]
            ];
        }
        if ($deliveryCharge) {
            $line_items[] = [
                'quantity' => 1, // Single delivery charge
                'price_data' => [
                    'unit_amount' => (int) ($deliveryCharge * 100), // Convert delivery charge to cents
                    'currency' => 'usd',
                    'product_data' => [
                        'name' => "Delivery Charge", // Name for the delivery charge
                    ]
                ]
            ];
        }
        $session = $this->stripe->checkout->sessions->create([
            'payment_method_types' => ['card'],
            'metadata' => [
                'service_message' => json_encode([
                    "user_id" => $user_id,
                    "order_id" => $order_id,
                    "unit_amount" => $amount,
                ])
            ],
            'line_items' =>
                $line_items,
            'mode' => 'payment',
            'success_url' => route('checkout.success'),
            'cancel_url' => route('checkout.cancel'),
        ]);
        Cache::set('stripe_checkout_id', $session->id);
        return $session;
    }

    // public function success()
    // {
    //     if (Cache::has('stripe_checkout_id')) {
    //         $stripe_checkout_id = Cache::get('stripe_checkout_id');
    //         $intenet_response = $this->stripe->checkout->sessions->retrieve($stripe_checkout_id, []);
    //         $service_message = json_decode($intenet_response->metadata->service_message);
    //         $user = User::find($service_message->user_id);
    //         if ($intenet_response->payment_status == "paid") {
    //             DB::beginTransaction();
    //             Cache::forget('stripe_checkout_id');
    //             $paymentLog = new PaymentLog();
    //             $paymentLog->paymentable_type = "App\Models\Order";
    //             $paymentLog->paymentable_id = $service_message->order_id;
    //             $paymentLog->stripe_checkout_id = $stripe_checkout_id;
    //             $paymentLog->currency = "usd";
    //             $paymentLog->amount = $service_message->unit_amount;
    //             $paymentLog->user_id = $service_message->user_id;
    //             $paymentLog->save();
    //             $order = Order::where('id', $service_message->order_id)->with('address', 'details')->first();
    //             $order->is_paid = 1;
    //             $order->rejection_reason = null;
    //             $order->status = Order::PENDING;
    //             $order->save();
    //             $this->subtractProductQuantity($order->details);
    //             DB::commit();
    //             if ($user != null && isset($user->first_name, $user->last_name, $user->id)) {
    //                 // Construct the title and body once
    //                 $title = $user->first_name . " " . $user->last_name . " created an order!";
    //                 $body = "User ID: " . $user->id;

    //                 // Create data array
    //                 $data = [
    //                     "user" => $user,  // Optionally, you can pass just the ID or a limited set of user info
    //                     "title" => $title,
    //                     "body" => $body,
    //                 ];

    //                 // If you're sending the notification or doing further processing with $data
    //                 // sendNotification($data);
    //             } else {
    //                 $address = OrderAddress::where('order_id', $order->id)->where('type', OrderAddress::CONTACT_INFORMATION)->first();
    //                 $title = $address->first_name . " " . $address->first_name . " as a guest created order !";
    //                 $body = $order->id;
    //                 $data = [
    //                     "user" => $address,
    //                     "title" => $title,
    //                     "body" => $body,
    //                 ];
    //                 Notification::route('mail', $address->email)->notify(new OrderNotification($order));
    //                 //send notification to guest user email
    //             }
    //             sendNotificationToAdmin($title, $body, $data);

    //             // Notification::route('firebase', $user->device_token)->notify(new PushNotification($title, $body));
    //             return view('front.payment-response', [
    //                 'status' => 'success',
    //                 'title' => 'Payment Successfull!',
    //                 'message' => 'Payment Successfull !',
    //             ]);

    //         }
    //         return view('front.payment-response', [
    //             'status' => 'danger',
    //             'title' => 'Error!',
    //             'message' => 'Something went wrong.'
    //         ]);
    //     }
    // }
    public function success()
    {
        if (Cache::has('stripe_checkout_id')) {
            $stripe_checkout_id = Cache::get('stripe_checkout_id');
            $intenet_response = $this->stripe->checkout->sessions->retrieve($stripe_checkout_id, []);
            $service_message = json_decode($intenet_response->metadata->service_message);
            $user = User::find($service_message->user_id);

            if ($intenet_response->payment_status == "paid") {
                DB::beginTransaction();
                Cache::forget('stripe_checkout_id');
                $paymentLog = new PaymentLog();
                $paymentLog->paymentable_type = "App\Models\Order";
                $paymentLog->paymentable_id = $service_message->order_id;
                $paymentLog->stripe_checkout_id = $stripe_checkout_id;
                $paymentLog->currency = "usd";
                $paymentLog->amount = $service_message->unit_amount;
                $paymentLog->user_id = $service_message->user_id;
                $paymentLog->save();

                $order = Order::where('id', $service_message->order_id)->with('address', 'details')->first();
                $order->is_paid = 1;
                $order->rejection_reason = null;
                $order->status = Order::PENDING;
                $order->save();

                $this->subtractProductQuantity($order->details);
                DB::commit();

                if ($user != null && isset($user->first_name, $user->last_name, $user->id)) {
                    // Construct the title and body once
                    $title = $user->first_name . " " . $user->last_name . " created an order!";
                    $body = "User ID: " . $user->id;

                    // Create data array
                    $data = [
                        "user" => $user,  // Optionally, you can pass just the ID or a limited set of user info
                        "title" => $title,
                        "body" => $body,
                    ];

                    // sendNotification($data);
                } else {
                    $address = OrderAddress::where('order_id', $order->id)->where('type', OrderAddress::CONTACT_INFORMATION)->first();
                    $title = $address->first_name . " " . $address->first_name . " as a guest created order !";
                    $body = $order->id;
                    $data = [
                        "user" => $address,
                        "title" => $title,
                        "body" => $body,
                    ];

                    Notification::route('mail', $address->email)->notify(new OrderNotification($order));
                    // Send notification to guest user email
                }

                sendNotificationToAdmin($title, $body, $data);

                // Redirect to the desired URL (e.g., https://bandeddating.com/)
                return redirect('https://bandeddating.com/')
                    ->with('status', 'Payment Successful!')
                    ->with('message', 'Your payment was successful!');
            }

            // Redirect to the same URL if payment was not successful
            return redirect('https://bandeddating.com/')
                ->with('status', 'Error!')
                ->with('message', 'Something went wrong.');
        }
    }

    public function subtractProductQuantity($orderDetails)
    {
        // Prepare the product quantities array from the order details
        $productQuantities = [];

        foreach ($orderDetails as $orderDetail) {
            // Assuming $orderDetail is an object with 'product' relation
            $productId = $orderDetail->product->id; // Get the product ID
            $quantity = $orderDetail->quantity; // Get the quantity to subtract

            // Only include products with a valid quantity
            if ($quantity > 0) {
                $productQuantities[$productId] = $quantity;
            }
        }

        // Check if there are products to update
        if (count($productQuantities) > 0) {
            // Update the product quantities in a single query
            Product::whereIn('id', array_keys($productQuantities))
                ->update([
                    'quantity' => DB::raw('CASE id ' .
                        collect($productQuantities)->map(function ($quantity, $id) {
                            return "WHEN {$id} THEN quantity - {$quantity}"; // Subtract the quantity from the current quantity
                        })->implode(' ') .
                        ' END')
                ]);
        }
    }

    public function cancel()
    {
        return view('front.payment-response', [
            'status' => 'danger',
            'title' => 'Payment Failed!',
            'message' => 'Something went wrong.'
        ]);
    }

}
