const fs = require("fs");
const path = require("path");
const express = require("express");
const cors = require("cors");
const http = require("https");
const redis = require("redis");
const axios = require("axios");
const PORT = 3119;

class Server {
    constructor(port) {
        this.port = port;
        this.app = express();
        this.redisClient = this.connectToRedis();
        this.httpServer = this.createServer();
        this.io = this.initSocket();
        this.setupMiddleware();
        this.setupSocketEvents();
    }

    connectToRedis() {
        const client = redis.createClient({
            url: "redis://127.0.0.1:6379", // replace with your Redis host and port
        });

        client.on("error", (err) => console.error("Redis Client Error", err));
        client.connect();
        return client;
    }

    createServer() {
        const credentials = {
            key: fs.readFileSync(
                path.resolve(
                    "/home/bandeddating/public_html/zeke-martin-backend/ssl/bandeddating.key"
                ),
                "utf8"
            ),
            cert: fs.readFileSync(
                path.resolve(
                    "/home/bandeddating/public_html/zeke-martin-backend/ssl/bandeddating.crt"
                ),
                "utf8"
            ),
            ca: fs.readFileSync(
                path.resolve(
                    "/home/bandeddating/public_html/zeke-martin-backend/ssl/bandeddating.ca"
                )
            ),
        };
        this.app.use(cors());
        const server = http.createServer(credentials, this.app);

        // const server = http.createServer(this.app);
        server.listen(this.port, () => {
            console.log(
                `\u001b[34mServer started on port: ${this.port}\u001b[0m`
            );
        });
        return server;
    }

    initSocket() {
        const io = require("./socketConfig/socket").init(this.httpServer);
        return io;
    }

    setupMiddleware() {
        // Any additional middleware can be set up here
    }

    setupSocketEvents() {
        this.io.on("connection", (socket) => {
            console.log("User Joined");

            socket.on("sendMessage", (data) => this.handleSendMessage(data));
            socket.on("updateLocation", (data) =>
                this.handleUpdateLocation(socket, data)
            );
            socket.on("getLocation", (userId) =>
                this.handleGetLocation(userId)
            );
            socket.on("sendRequest", (data) => this.handleSendRequest(data));
            socket.on("sendLocation", (data) => this.handleSendLocation(data));
            socket.on("disconnect", () => this.handleDisconnect(socket));
        });
    }

    handleSendMessage(data) {
        console.log("Message Received");
        this.io.emit(data.chat_id, data);
        this.io.emit("chatListEvent", data);
    }

    handleSendRequest(data) {
        console.log("Handle Request Send");
        console.log(data);
        console.log(typeof data);
        console.log(data?.user_id);
        this.io.emit("request." + data?.user_id, data);
    }

    handleSendLocation(data) {
        // console.log("Handle Request Send");
        // console.log(data);
        // console.log(typeof data);
        // console.log(data?.user_id);
        this.io.emit("location." + data?.user_id, data);
    }

    // Haversine Formula to calculate distance between two lat/lng points in feet
    calculateDistance(location1, location2) {
        const lat1 = location1.latitude;
        const lon1 = location1.longitude;
        const lat2 = location2.latitude;
        const lon2 = location2.longitude;

        const R = 6371e3; // Earth radius in meters
        const φ1 = (lat1 * Math.PI) / 180;
        const φ2 = (lat2 * Math.PI) / 180;
        const Δφ = ((lat2 - lat1) * Math.PI) / 180;
        const Δλ = ((lon2 - lon1) * Math.PI) / 180;

        const a =
            Math.sin(Δφ / 2) * Math.sin(Δφ / 2) +
            Math.cos(φ1) * Math.cos(φ2) * Math.sin(Δλ / 2) * Math.sin(Δλ / 2);
        const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
        const distanceInMeters = R * c; // distance in meters

        const distanceInFeet = distanceInMeters * 3.28084; // convert to feet
        return distanceInFeet;
    }

    async handleUpdateLocation(socket, { user_id, location }) {
        try {
            await this.redisClient.hSet(
                "userLocations",
                user_id,
                JSON.stringify(location)
            );

            await this.redisClient.hSet("userSessions", socket.id, user_id);

            // get all user locations and check if current user is in 500m radius of each user
            // Retrieve all user locations from Redis
            const allLocations = await this.redisClient.hGetAll(
                "userLocations"
            );
            if (!allLocations) {
                console.error(
                    "Error: Redis returned null or undefined for userLocations"
                );
                return;
            }

            // Iterate over each user's location and check proximity
            let userInRadius = [];
            for (const [otherUserId, otherLocationJson] of Object.entries(
                allLocations
            )) {
                console.log("other user id", otherUserId);

                if (otherUserId == user_id) {
                } else {
                    const otherLocation = JSON.parse(otherLocationJson);
                    const distance = this.calculateDistance(
                        location,
                        otherLocation
                    ); // Implement this function

                    // If the distance is within 500m
                    if (distance <= 500) {
                        console.log("500 feet radius", userInRadius);
                        userInRadius.push(otherUserId);
                    }
                }
            }
            console.log("api call");
            console.log("api call", userInRadius);
            console.log("user_id", user_id);
            if (userInRadius.length > 0) {
                let response = await axios.post(
                    "https://bandeddating.com/zeke-martin-backend/api/user/notification/firebase/send",
                    {
                        user_id: user_id,
                        userInRadius: userInRadius,
                    }
                );
                console.log(response.data.message);
            }
            console.log("api call end");

            this.io.emit(`userLocation${user_id}`, { user_id, location });
            console.log(`Updated location for ${user_id}:`, location);
        } catch (error) {
            console.error("Error updating location:", error);
        }
    }

    async handleGetLocation(user_id) {
        if (typeof userId !== "string" || !user_id) {
            console.error("Invalid userId:", user_id);
            return;
        }

        const location = await this.redisClient.hGet("userLocations", user_id);

        // get all user locations and check if current user is in 500m radius of each user
        // Retrieve all user locations from Redis
        const allLocations = await this.redisClient.hGetAll("userLocations");
        console.log("handleGetLocation all", allLocations);

        // Iterate over each user's location and check proximity
        for (const [otherUserId, otherLocationJson] of Object.entries(
            allLocations
        )) {
            if (otherUserId === user_id) continue; // Skip current user's own location

            const otherLocation = JSON.parse(otherLocationJson);
            const distance = this.calculateDistance(location, otherLocation); // Implement this function

            console.log("distance", distance);
            // If the distance is within 500m
            let userInRadius = [];
            if (distance <= 500) {
                console.log("500m radius", userInRadius);
                userInRadius.push(otherUserId);
            }

            console.log("api call");
            if (userInRadius.length > 0) {
                let response = await axios.post(
                    "https://bandeddating.com/zeke-martin-backend/api/user/notification/firebase/send",
                    {
                        user_id: user_id,
                        userInRadius: userInRadius,
                    }
                );
                console.log(response.data.message);
            }
            console.log("api call end");
        }

        console.log(`Get location for ${user_id}:`, location);
        this.io.emit(`userLocation${user_id}`, {
            userId,
            location: location ? JSON.parse(location) : null,
        });
    }

    async handleDisconnect(socket) {
        try {
            // Ensure Redis client is connected
            if (!this.redisClient.isOpen) {
                console.error("Redis client is not connected");
                return;
            }

            // Get user session from Redis
            const userId = await this.redisClient.hGet(
                "userSessions",
                socket.id
            );

            // If no session is found, log and exit
            if (!userId) {
                console.log("No user session found for socket id:", socket.id);
                return;
            }

            // Get location data for the user
            const locationData = await this.redisClient.hGet(
                "userLocations",
                userId
            );

            let location;
            if (locationData) {
                location = JSON.parse(locationData);
            }

            // Delete user session
            // await this.redisClient.hDel("userSessions", socket.id);
            // console.log("User session deleted for:", socket.id);
            console.log("User info:", userId);

            // If location exists, send the location update
            if (location) {
                console.log(
                    "User location:",
                    location.latitude,
                    location.longitude
                );

                try {
                    await axios.post(
                        "https://bandeddating.com/zeke-martin-backend/api/location/update",
                        {
                            user_id: userId,
                            longitude: location.longitude,
                            latitude: location.latitude,
                        }
                    );
                    console.log("Location sent successfully.");
                } catch (axiosError) {
                    console.error(
                        "Error sending location update:",
                        axiosError.message,
                        axiosError.response?.data
                    );
                }
            }
        } catch (error) {
            console.error("Error during disconnect:", error);
        }
    }
}

// Initialize server
new Server(PORT);
