import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { faFilter, faSearch } from "@fortawesome/free-solid-svg-icons";
import React, { useCallback, useEffect, useState } from "react";
import Button from "react-bootstrap/Button";
import Offcanvas from "react-bootstrap/Offcanvas";
import "./style.css";
import CustomButton from "../CustomButton";
import { Select } from "../Select";
import { sorting } from "../../Config/TableStatus";
import debounce from "lodash.debounce";
import SiteInput from "../Input/input";
import { Dropdown } from "react-bootstrap";

const CustomFilters = (props) => {
  const [formData, setFormData] = useState({});

  useEffect(() => {
    setFormData(props.filters);
  }, [props.filters]);

  const debouncedSetFilters = useCallback(
    debounce((updatedFormData) => {
      props.setFilters(updatedFormData);
    }, 500),
    [props.setFilters]
  );

  const handleChange = (name, value) => {
    const updatedFormData = {
      ...formData,
      [name]: value,
    };

    setFormData(updatedFormData);

    if (name === "search") {
      debouncedSetFilters(updatedFormData);
    } else if (name === "per_page") {
      props.setFilters(updatedFormData);
    }
  };

  const handleInputChange = (event) => {
    const { name, value } = event.target;
    handleChange(name, value);
  };

  const handleSelectChange = (name, value) => {
    handleChange(name, value);
  };

  const handleApply = () => {
    props.setFilters(formData);
  };

  const handleClear = () => {
    const clearedFilters = {
      page: 1,
      per_page: 10,
    };
    setFormData(clearedFilters);
    props.setFilters(clearedFilters);
  };

  return (
    <>
      <div className="tableFilters">
        <div className="d-flex justify-content-between gap-3 flex-wrap align-items-center">
          <div className="d-flex align-items-center gap-3">
            <Select
              className="mainInput entries-input w-auto "
              value={formData?.per_page}
              name="per_page"
              onChange={(value) => handleSelectChange("per_page", value)}
              label="Show"
              labelclass="secondaryLabel mb-0"
            >
              {sorting}
            </Select>
          </div>

          <div className="d-flex align-items-center justify-content-end gap-2">
            <div className="filterWrapper d-md-flex align-items-baseline mb-0">
              <div className="searchWrapper">
                <SiteInput
                  type="text"
                  placeholder="Search..."
                  name="search"
                  className="searchInput mainInput"
                  value={formData?.search || ""}
                  onChange={handleInputChange}
                />
                <button className="searchButton notButton">
                  <FontAwesomeIcon icon={faSearch} />
                </button>
              </div>
            </div>
            <Dropdown className="filter-dropdown">
              <Dropdown.Toggle className="btn_filter">
                <FontAwesomeIcon icon={faFilter} />
              </Dropdown.Toggle>

              <Dropdown.Menu align={'end'}>
                <h2 className="mainTitle p-md text-center">Filters</h2>
                <div className="mb-3">
                  {props?.dateFilters?.map(({ title, from, to }, index) => (
                    <div className="filterWrapper" key={index}>
                      <label className="filterLabel w-100 mb-2">Filter By: {title}</label>
                      <input
                        type="date"
                        placeholder="From"
                        name={from}
                        className="w-100 mb-3 mainInput"
                        value={formData[from] || ""}
                        onChange={handleInputChange}
                      />
                      <input
                        type="date"
                        placeholder="To"
                        name={to}
                        className="w-100 mainInput"
                        value={formData[to] || ""}
                        onChange={handleInputChange}
                      />
                    </div>
                  ))}
                </div>

                <div className="mb-3">
                  {props?.selectOptions?.map((option, index) => (
                    <div className="mb-3" key={index}>
                      {option ? (
                        <Select
                          name={option.title}
                          className="mainInput w-100"
                          labelclass="mb-2"
                          value={formData[option.title] || ""}
                          onChange={(value) => handleChange(option.title, value)}
                          label={`Filter by ${option?.title}`}
                        >
                          {option?.options}
                        </Select>
                      ) : null}
                    </div>
                  ))}
                </div>
                <div className="mb-3 mt-4 d-flex align-items-center justify-content-center gap-3 flex-wrap">
                  <CustomButton
                    onClick={handleApply}
                    type="button"
                    variant="btnBlueBG"
                    text="Apply"
                    className="mb-3"
                  />
                  <CustomButton
                    onClick={handleClear}
                    type="button"
                    variant="btnBlueBor"
                    className="mb-3"
                    text="Clear"
                  />
                </div>
              </Dropdown.Menu>
            </Dropdown>
          </div>
        </div>
      </div>
    </>
  );
};

export default CustomFilters;
