import { useState, useEffect } from "react";
import "./style.css";
import NotificationCard from "../../NotificationCard";
import { getAll, post } from "../../../Services/Api";
import { dateFormat, humanReadable } from "../../../Utils/helper";
import { images } from "../../../Assets";
import { Select } from "../../Select";
import { notificationOptions } from "../../../Config/TableStatus";
import LoadingSpinner from "../../Loader";
import withPagination from "../../../HOC/withPagination";
import { notificationsData } from "../../../Config/data";
import { UserLayout } from "../Layout";
import CustomPagination from "../../CustomPagination";

const UserConsolidatedNotifications = ({
  apiEndpoint,
  filters,
  setFilters,
  pagination,
  updatePagination,
}) => {
  const [notifications, setNotifications] = useState([]);
  const [loading, setLoading] = useState(true);
  const [selectedStatus, setSelectedStatus] = useState("");

  const fetchNotifications = async () => {
    setLoading(true);
    const response = await getAll(`${apiEndpoint}${selectedStatus ? `?status=${selectedStatus}` : ""}`, filters);
    if (response.status) {
      const { total, per_page, current_page, to } = response.meta;
      setNotifications(response.data);
      updatePagination({
        showData: to,
        currentPage: current_page,
        totalRecords: total,
        totalPages: Math.ceil(total / per_page),
      });
      setLoading(false);
    }
    setLoading(false);
  };
  useEffect(() => {
    
    fetchNotifications();
  }, [filters, selectedStatus]);

  const handleNotificationAction = async (val) => {
    const response = await post(`/user/notification/read`, { "uuid": val });
    if (response.status) {
      fetchNotifications();
    }
  };

  return (
    <UserLayout pageTitle="Notifications">
      <section className="site-notifications py-sm-5 py-4">
        <div className="container">
          <div className="row mb-3">
            <div className="col-12">
              <h4 className="heavy">Notifications</h4>
            </div>
          </div>
          <div className="row mb-2">
            <div className="col-12">
              <div className="d-flex align-items-center justify-content-between">
                <div className="d-flex align-items-center">
                  <Select
                    className="notificationSelect"
                    value={selectedStatus}
                    onChange={setSelectedStatus}
                    label="Show"
                    labelclass="pe-3"
                  >
                    {notificationOptions}
                  </Select>
                </div>
                {/* <button className="notButton">Mark All As Read</button> */}
              </div>
            </div>
          </div>
          <div className="row">
            {loading ? (
              <LoadingSpinner />
            ) : (
              notifications.map((notification) => (
                <div className="col-12" key={notification.id}>
                  <NotificationCard
                    id={notification.id}
                    text={notification?.body}
                    name={notification?.title}
                    image={images.notiImg}
                    date={dateFormat(notification?.created_at)}
                    time={humanReadable(notification?.created_at)}
                    read={notification?.read_at}
                    notiRead={notification?.read_at}
                    onClick={handleNotificationAction}
                  />
                </div>
              ))
            )}
          </div>
          <div className="row">
            <div className="col-12">
              <CustomPagination pagination={pagination} setFilters={setFilters} />
            </div>
          </div>
        </div>
      </section>
    </UserLayout>
  );
};

export default withPagination(UserConsolidatedNotifications);
