import React, { useState, useEffect } from "react";
import './index.css';

const QuantityInput = ({ min = 1, max = 100, step = 1, initialQuantity = 1, onChange }) => {
  const [quantity, setQuantity] = useState(initialQuantity);

  useEffect(() => {
    setQuantity(initialQuantity);
  }, [initialQuantity]);

  const handleIncrement = () => {
    setQuantity(prevQuantity => {
      const newQuantity = Math.min(parseInt(prevQuantity) + parseInt(step), parseInt(max));
      if (newQuantity !== prevQuantity) {
        onChange && onChange(newQuantity);
      }
      return newQuantity;
    });
  };

  const handleDecrement = () => {
    setQuantity(prevQuantity => {
      const newQuantity = Math.max(parseInt(prevQuantity) - parseInt(step), parseInt(min));
      if (newQuantity !== prevQuantity) {
        onChange && onChange(newQuantity);
      }
      return newQuantity;
    });
  };

  const handleChange = (e) => {
    let value = parseInt(e.target.value, 10);

    // Ensure the value is within min and max limits
    if (isNaN(value)) {
      value = min;
    } else if (value < min) {
      value = min;
    } else if (value > max) {
      value = max;
    }

    setQuantity(value);
    onChange && onChange(value);
  };

  return (
    <div className="quantity-input">
      <button
        className="decrement-btn"
        onClick={handleDecrement}
        disabled={quantity <= min}
        aria-label="Decrease quantity"
      >
        -
      </button>
      <input
        type="number"
        value={quantity}
        onChange={handleChange}
        min={min}
        max={max}
        step={step}
        aria-label="Quantity"
      />
      <button
        className="increment-btn"
        onClick={handleIncrement}
        disabled={quantity >= max}
        aria-label="Increase quantity"
      >
        +
      </button>
    </div>
  );
};

export default QuantityInput;
