import { useState, useEffect } from "react";
import { DashboardLayout } from "../../../Components/AdminLayout/DashboardLayout";
import CustomTable from "../../../Components/CustomTable";
import { dateFormat, getStatus, getStatusByText, serialNum } from "../../../Utils/helper";
import { categoryHeaders } from "../../../Config/TableHeaders"; // Table Headers
import { generalStatus, ProductStatus } from "../../../Config/TableStatus"; // Filter Status
import { getAll, post } from "../../../Services/Api"; //Api Service
import TableDropdown from "../../../Components/TableDropdown";
import "./style.css";
import { useFormStatus } from "../../../Hooks/useFormStatus";
import withFilters from "../../../HOC/withFilters ";
import withModal from "../../../HOC/withModal";
import { categoryManagementData } from "../../../Config/data";
import CustomButton from "../../../Components/CustomButton";
import { useNavigate } from "react-router-dom";

const CategoryManagement = ({
  showModal,
  filters,
  setFilters,
  pagination,
  updatePagination,
}) => {
  const [categoryData, setCategoryData] = useState([]);
  const { isSubmitting, startSubmitting, stopSubmitting } = useFormStatus();

  const fetchCategories = async () => {
    startSubmitting(true);
    const url = `/admin/category/categories`;
    const response = await getAll(url, filters);
    if (response.status) {
      const { total, per_page, current_page, to } = response.meta;
      setCategoryData(response.data);
      updatePagination({
        showData: to,
        currentPage: current_page,
        totalRecords: total,
        totalPages: Math.ceil(total / per_page),
      });
      stopSubmitting(false);
    }
    stopSubmitting(false);
  };

  const confirmPopup = (id, status) => {
    console.log(status);

    showModal(
      `Are you sure you want to ${status === 1 ? "Inactivate" : "Activate"
      } this Category?`,
      () => onConfirm(id, status)
    );
  };

  const onConfirm = async (id, status) => {

    const response = await post(`/admin/category/${id}/status`);
    if (response.status) {
      await fetchCategories();
      showModal(
        `Category ${response?.data?.status === "active" ? "activated" : "inactivated"
        } successfully`, //heading
        null, //action
        true //sucess
      );
    }
  };

  useEffect(() => {
    fetchCategories();
  }, [filters]);
  const navigate = useNavigate();
  return (
    <DashboardLayout pageTitle="Category Management">
      <div className="container-fluid">
        <div className="row">
          <div className="col-12">
            <div className="dashCard">
              <div className="row mb-5">
                <div className="col-12 d-flex align-items-center justify-content-between">
                  <h2 className="mainTitle mb-0">Category Management</h2>
                  <CustomButton text="Add Category" onClick={() => navigate('/admin/category-management/add/')} />
                </div>
              </div>
              <div className="statsCard">
                <div className="row mb-3">
                  <div className="col-12">
                    <CustomTable
                      filters={filters}
                      setFilters={setFilters}
                      loading={isSubmitting}
                      headers={categoryHeaders}
                      pagination={pagination}
                      // if you want multiple date filters
                      dateFilters={[
                        {
                          from: "fromDate",
                          to: "toDate",
                        },
                      ]}
                      // if you want multiple select filters
                      selectOptions={[
                        {
                          title: "status",
                          options: ProductStatus,
                        },
                      ]}
                    >
                      <tbody>
                        {categoryData?.map((item, index) => (
                          <tr key={item?.id}>
                            <td>
                              {serialNum(
                                (filters.page - 1) * filters.per_page + index + 1
                              )}
                            </td>
                            <td>{item?.name}</td>
                            <td>{dateFormat(item?.created_at)}</td>
                            <td>{item?.total_products}</td>
                            <td
                              className={getStatusByText(item?.status).className}
                            >
                              {getStatusByText(item?.status).text}
                            </td>
                            <td>
                              <TableDropdown
                                view
                                itemId={item?.id}
                                linkPath="/admin/category-management/details"
                                handleStatusChange={confirmPopup}
                                statusDetail={item?.status}
                              />
                            </td>
                          </tr>
                        ))}
                      </tbody>
                    </CustomTable>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </DashboardLayout>
  );
};

//Additional Filters
const additionalFilters = {
  type: "",
  expiryFromDate: "",
  expiryToDate: "",
};

export default withModal(withFilters(CategoryManagement, additionalFilters));
// Bind with HOC
// withModal is for Pop Up
// withFilters is for Pagination and default filters
// withPagination, if you want only pagination not filters
// additionalFilters, if you want extra filters on specific pages
