import { useState, useEffect } from "react";
import { DashboardLayout } from "../../../Components/AdminLayout/DashboardLayout";
import CustomTable from "../../../Components/CustomTable";
import { dateFormat, getStatus, serialNum } from "../../../Utils/helper";
import { deliveryChargesHeaders } from "../../../Config/TableHeaders"; // Table Headers
import { generalStatus } from "../../../Config/TableStatus"; // Filter Status
import { getAll, post } from "../../../Services/Api"; //Api Service
import TableDropdown from "../../../Components/TableDropdown";
import "./style.css";
import { useFormStatus } from "../../../Hooks/useFormStatus";
import withFilters from "../../../HOC/withFilters ";
import withModal from "../../../HOC/withModal";
import { deliveryChargesManagementData } from "../../../Config/data";
import { Form, Formik } from "formik";
import FormikField from "../../../Components/FormikField";
import CustomButton from "../../../Components/CustomButton";
import { addDeliviryChargeValidationSchema } from "../../../Config/Validations";

const DeliveryChargesManagement = ({
  showModal,
  filters,
  setFilters,
  pagination,
  updatePagination,
}) => {
  const [deliveryChargesData, setDeliveryChargesData] = useState([]);
  const { isSubmitting, startSubmitting, stopSubmitting } = useFormStatus();

  const fetchDeliveryCharges = async () => {
    startSubmitting(true);
    const url = `/admin/deliveryCharge/deliveryCharges`;
    const response = await getAll(url, filters);
    if (response.status) {
      const { total, per_page, current_page, to } = response.meta;
      setDeliveryChargesData(response.data);
      updatePagination({
        showData: to,
        currentPage: current_page,
        totalRecords: total,
        totalPages: Math.ceil(total / per_page),
      });
      stopSubmitting(false);
    }
    stopSubmitting(false);
  };

  useEffect(() => {
    fetchDeliveryCharges();
  }, [filters]);
  const create = async (values, { resetForm }) => {
    startSubmitting();
    const response = await post("/admin/deliveryCharge/deliveryCharge", values);
    if (response.status) {
      showModal(
        `Delivery Charges Has Been Updated Successfully`, //heading
        null, //action
        true //sucess
      )
      fetchDeliveryCharges();
      resetForm();
    }
    stopSubmitting();
  };
  return (
    <DashboardLayout pageTitle="Delivery Charges Management">
      <div className="container-fluid">
        <div className="row">
          <div className="col-12">
            <div className="dashCard">
              <div className="row mb-5">
                <div className="col-md-6">
                  <h2 className="mainTitle mb-0">Delivery Charges Management</h2>
                </div>
              </div>
              <div className="statsCard">
                <div className="row mb-3">
                  <div className="col-12">
                    <Formik
                      initialValues={{ delivery_charges: "" }}
                      validationSchema={addDeliviryChargeValidationSchema}
                      onSubmit={create}
                    >
                      {({ isSubmitting }) => (
                        <Form>
                          <div className="details-wrapper">
                            <div className="row">
                              <FormikField
                                span
                                name="delivery_charges"
                                label="Delivery Charges"
                                labelClass="mainLabel"
                                placeholder="Enter Delivery Charges"
                                className="mainInput"
                              />
                            </div>
                          </div>
                          <div className="row">
                            <div className="col-md-12">
                              <CustomButton
                                variant="btnBlueBG"
                                className="px-5"
                                text="Update"
                                pendingText="Submitting..."
                                isPending={isSubmitting}
                                type="submit"
                              />
                            </div>
                          </div>
                        </Form>
                      )}
                    </Formik>
                    <CustomTable
                      filters={filters}
                      setFilters={setFilters}
                      loading={isSubmitting}
                      headers={deliveryChargesHeaders}
                      pagination={pagination}
                      // if you want multiple date filters
                      dateFilters={[
                        {
                          from: "fromDate",
                          to: "toDate",
                        },
                      ]}
                    >
                      <tbody>
                        {deliveryChargesData?.map((item, index) => (
                          <tr key={item?.id}>
                            <td>
                              {serialNum(
                                (filters.page - 1) * filters.per_page + index + 1
                              )}
                            </td>
                            <td>${item?.amount}</td>
                            <td>{dateFormat(item?.created_at)}</td>
                          </tr>
                        ))}
                      </tbody>
                    </CustomTable>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </DashboardLayout>
  );
};

//Additional Filters
const additionalFilters = {
  type: "",
  expiryFromDate: "",
  expiryToDate: "",
};

export default withModal(withFilters(DeliveryChargesManagement, additionalFilters));
// Bind with HOC
// withModal is for Pop Up
// withFilters is for Pagination and default filters
// withPagination, if you want only pagination not filters
// additionalFilters, if you want extra filters on specific pages
