import { useEffect, useState } from "react";
import { useNavigate } from "react-router-dom";
import { DashboardLayout } from "../../../Components/AdminLayout/DashboardLayout";
import CustomButton from "../../../Components/CustomButton";
import CustomTable from "../../../Components/CustomTable";
import TableDropdown from "../../../Components/TableDropdown";
import { bannerHeaders, eventHeaders } from "../../../Config/TableHeaders"; // Table Headers
import withFilters from "../../../HOC/withFilters ";
import { useFormStatus } from "../../../Hooks/useFormStatus";
import { getAll } from "../../../Services/Api"; //Api Service
import { dateFormat, serialNum } from "../../../Utils/helper";

const EventManagement = ({
  filters,
  setFilters,
  pagination,
  updatePagination,
}) => {
  const [eventData, setEventData] = useState([]);
  const { isSubmitting, startSubmitting, stopSubmitting } = useFormStatus();

  const fetchBanners = async () => {
    startSubmitting(true);
    const url = `/admin/event/events`;
    const response = await getAll(url, filters);
    if (response.status) {
      const { total, per_page, current_page, to } = response.meta;
      setEventData(response.data);
      updatePagination({
        showData: to,
        currentPage: current_page,
        totalRecords: total,
        totalPages: Math.ceil(total / per_page),
      });
      stopSubmitting(false);
    }
    stopSubmitting(false);
  };

  useEffect(() => {
    fetchBanners();
  }, [filters]);
  const navigate = useNavigate();
  return (
    <DashboardLayout pageTitle="Event Management">
      <div className="container-fluid">
        <div className="row">
          <div className="col-12">
            <div className="dashCard">
              <div className="row mb-5">
                <div className="col-12 d-flex align-items-center gap-3 flex-wrap justify-content-between">
                  <h2 className="mainTitle mb-0">Event Management</h2>
                  <CustomButton onClick={() => navigate('/admin/event-management/add')} text="Add Event" />
                </div>
              </div>
              <div className="statsCard">
                <div className="row mb-3">
                  <div className="col-12">
                    <CustomTable
                      filters={filters}
                      setFilters={setFilters}
                      loading={isSubmitting}
                      headers={eventHeaders}
                      pagination={pagination}
                      // if you want multiple date filters
                      dateFilters={[
                        {
                          title: 'Posted Date',
                          from: "fromDate",
                          to: "toDate",
                        },
                        {
                          title: 'Expiry Date',
                          from: "expiryFromDate",
                          to: "expiryToDate",
                        },
                      ]}
                    >
                      <tbody>
                        {eventData?.map((item, index) => (
                          <tr key={item?.id}>
                            <td>
                              {serialNum(
                                (filters.page - 1) * filters.per_page + index + 1
                              )}
                            </td>
                            <td>{item?.name}</td>
                            <td>{dateFormat(item?.created_at)}</td>
                            <td>{dateFormat(item?.date)}</td>
                            <td>{item?.address}</td>
                            <td>
                              <TableDropdown
                                view
                                itemId={item?.id}
                                linkPath="/admin/event-management/details"
                                statusDetail={item?.status_detail}
                              />
                            </td>
                          </tr>
                        ))}
                      </tbody>
                    </CustomTable>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </DashboardLayout>
  );
};

export default withFilters(EventManagement);
// Bind with HOC
// withModal is for Pop Up
// withFilters is for Pagination and default filters
// withPagination, if you want only pagination not filters
// additionalFilters, if you want extra filters on specific pages
