import { useState, useEffect } from "react";
import { DashboardLayout } from "../../../Components/AdminLayout/DashboardLayout";
import CustomTable from "../../../Components/CustomTable";
import "./style.css";
import { Link } from "react-router-dom";
import { faEye } from "@fortawesome/free-solid-svg-icons";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { dateFormat, serialNum } from "../../../Utils/helper";
import { getAll } from "../../../Services/Api";
import { feedbackHeaders } from "../../../Config/TableHeaders";
import { feedbackStatus, feedbackTypes } from "../../../Config/TableStatus";
import { useFormStatus } from "../../../Hooks/useFormStatus";
import withFilters from "../../../HOC/withFilters ";
const FeedbacksManagement = ({
  filters,
  setFilters,
  pagination,
  updatePagination,
}) => {
  const [feedbackData, setFeedbackData] = useState([]);
  const { isSubmitting, startSubmitting, stopSubmitting } = useFormStatus();

  const getFeedback = async () => {
    startSubmitting(true);
    const url = `/admin/feedbacks`;
    const response = await getAll(url, filters);
    if (response.status) {
      const { total, per_page, current_page, to } = response.meta;
      setFeedbackData(response.data);
      updatePagination({
        showData: to,
        currentPage: current_page,
        totalRecords: total,
        totalPages: Math.ceil(total / per_page),
      });
      stopSubmitting(false);
    }
    stopSubmitting(false);
  };

  useEffect(() => {
    getFeedback();
  }, [filters]);

  return (
    <>
      <DashboardLayout pageTitle="Feedback Management">
        <div className="container-fluid">
          <div className="row">
            <div className="col-12">
              <div className="dashCard">
                <div className="row mb-5">
                  <div className="col-md-6">
                    <h2 className="mainTitle mb-0">Feedback Management</h2>
                  </div>
                </div>

                <div className="row mb-3">
                  <div className="col-12">
                    <CustomTable
                      loading={isSubmitting}
                      headers={feedbackHeaders}
                      filters={filters}
                      setFilters={setFilters}
                      pagination={pagination}
                      dateFilters={[
                        {
                          title: "Posted Date",
                          from: "fromDate",
                          to: "toDate",
                        },
                      ]}
                      selectOptions={[
                        {
                          title: "type",
                          options: feedbackStatus,
                        },
                        {
                          title: "status",
                          options: feedbackTypes,
                        },
                      ]}
                    >
                      <tbody>
                        {feedbackData?.map((item, index) => (
                          <tr key={index}>
                            <td>
                              {serialNum(
                                (filters.page - 1) * filters.per_page +
                                index +
                                1
                              )}
                            </td>
                            <td>{item?.full_name}</td>
                            <td>{item.email}</td>
                            <td>{dateFormat(item.created_at)}</td>
                            <td>{item.subject}</td>
                            <td
                              className={
                                item.status === "Resolved"
                                  ? "greenColor"
                                  : "redColor"
                              }
                            >
                              {item.status}
                            </td>
                            <td>
                              <Link
                                to={`/admin/feedbacks-management/${item.id}`}
                                className="tableAction"
                              >
                                <FontAwesomeIcon icon={faEye} />
                              </Link>
                            </td>
                          </tr>
                        ))}
                      </tbody>
                    </CustomTable>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </DashboardLayout>
    </>
  );
};
export default withFilters(FeedbacksManagement);
