import { useState, useEffect } from "react";
import { DashboardLayout } from "../../../Components/AdminLayout/DashboardLayout";
import CustomTable from "../../../Components/CustomTable";
import { dateFormat, serialNum } from "../../../Utils/helper";
import { orderHeaders } from "../../../Config/TableHeaders"; // Table Headers
import { deliveryStatus } from "../../../Config/TableStatus"; // Filter Status
import { getAll, post } from "../../../Services/Api"; //Api Service
import TableDropdown from "../../../Components/TableDropdown";
import "./style.css";
import { useFormStatus } from "../../../Hooks/useFormStatus";
import withFilters from "../../../HOC/withFilters ";
import withModal from "../../../HOC/withModal";
import { orderManagementData } from "../../../Config/data";

const OrderManagement = ({
  showModal,
  filters,
  setFilters,
  pagination,
  updatePagination,
}) => {
  const [orderData, setOrderData] = useState([]);
  const { isSubmitting, startSubmitting, stopSubmitting } = useFormStatus();

  const fetchOrders = async () => {
    startSubmitting(true);
    const url = `/admin/order/orders`;
    const response = await getAll(url, filters);
    if (response.status) {
      const { total, per_page, current_page, to } = response.meta;
      setOrderData(response.data);
      updatePagination({
        showData: to,
        currentPage: current_page,
        totalRecords: total,
        totalPages: Math.ceil(total / per_page),
      });
      stopSubmitting(false);
    }
  };

  useEffect(() => {
    fetchOrders();
  }, [filters]);

  return (
    <DashboardLayout pageTitle="Order Management">
      <div className="container-fluid">
        <div className="row">
          <div className="col-12">
            <div className="dashCard">
              <div className="row mb-5">
                <div className="col-md-6">
                  <h2 className="mainTitle mb-0">Order Management</h2>
                </div>
              </div>
              <div className="statsCard">
                <div className="row mb-3">
                  <div className="col-12">
                    <CustomTable
                      filters={filters}
                      setFilters={setFilters}
                      loading={isSubmitting}
                      headers={orderHeaders}
                      pagination={pagination}
                      // if you want multiple date filters
                      dateFilters={[
                        {
                          from: "fromDate",
                          to: "toDate",
                        },
                      ]}
                      // if you want multiple select filters
                      selectOptions={[
                        {
                          title: "status",
                          options: deliveryStatus,
                        },
                      ]}
                    >
                      <tbody>
                        {orderData?.map((item, index) => (
                          <tr key={item?.id}>
                            <td>
                              {serialNum(
                                (filters.page - 1) * filters.per_page + index + 1
                              )}
                            </td>
                            <td>#{item?.id}</td>
                            <td>{item?.user?.full_name ? item?.user?.full_name : "Guest"}</td>
                            <td>{dateFormat(item?.created_at)}</td>
                            <td>${item?.total_price}</td>
                            <td>{item?.status}</td>
                            <td>
                              <TableDropdown
                                view
                                itemId={item?.id}
                                linkPath="/admin/order-management/details"
                              />
                            </td>
                          </tr>
                        ))}
                      </tbody>
                    </CustomTable>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </DashboardLayout>
  );
};

//Additional Filters
const additionalFilters = {
  type: "",
  expiryFromDate: "",
  expiryToDate: "",
};

export default withModal(withFilters(OrderManagement, additionalFilters));
// Bind with HOC
// withModal is for Pop Up
// withFilters is for Pagination and default filters
// withPagination, if you want only pagination not filters
// additionalFilters, if you want extra filters on specific pages
