import { useState, useEffect } from "react";
import { DashboardLayout } from "../../../Components/AdminLayout/DashboardLayout";
import CustomTable from "../../../Components/CustomTable";
import { dateFormat, getStatus, getStatusByText, serialNum } from "../../../Utils/helper";
import { productHeaders } from "../../../Config/TableHeaders"; // Table Headers
import { generalStatus, ProductStatus } from "../../../Config/TableStatus"; // Filter Status
import { getAll, post } from "../../../Services/Api"; //Api Service
import TableDropdown from "../../../Components/TableDropdown";
import "./style.css";
import { useFormStatus } from "../../../Hooks/useFormStatus";
import withFilters from "../../../HOC/withFilters ";
import withModal from "../../../HOC/withModal";
import { categoryManagementData, productsManagementData } from "../../../Config/data";
import CustomButton from "../../../Components/CustomButton";
import { useNavigate } from "react-router-dom";

const ProductManagement = ({
  showModal,
  filters,
  setFilters,
  pagination,
  updatePagination,
}) => {
  const [productData, setProductData] = useState([]);
  const [categoriesData, setCategoriesData] = useState([]);
  const { isSubmitting, startSubmitting, stopSubmitting } = useFormStatus();

  const fetchData = async () => {
    startSubmitting(true);
    const url = `/admin/product/products`;
    const response = await getAll(url, filters);
    if (response.status) {
      const { total, per_page, current_page, to } = response.meta;
      setProductData(response.data);
      updatePagination({
        showData: to,
        currentPage: current_page,
        totalRecords: total,
        totalPages: Math.ceil(total / per_page),
      });
      stopSubmitting(false);
    }
    stopSubmitting(false);

  };
  const fetchCategories = async () => {
    startSubmitting(true);
    const url = `/admin/category/categories/withoutPagination`;
    const response = await getAll(url);
    if (response.status) {
      setCategoriesData(response.data);
      stopSubmitting(false);
    }
    stopSubmitting(false);

  };
  const confirmPopup = (id, status) => {
    console.log("status", status);

    showModal(
      `Are you sure you want to ${status === 1 ? "Inactivate" : "Activate"
      } this Product?`,
      () => onConfirm(id, status)
    );
  };

  const onConfirm = async (id, status) => {

    const response = await post(`/admin/product/${id}/status`);
    if (response.status) {
      await fetchData();
      console.log("response status", response?.data?.status);

      showModal(
        `Product ${response?.data?.status !== "active" ? "inactivated" : "activated"
        } successfully`, //heading
        null, //action
        true //sucess
      );
    }
  };

  useEffect(() => {
    fetchData();
  }, [filters]);
  useEffect(() => {
    fetchCategories();
  }, []);
  const navigate = useNavigate();
  let categoryOptions = categoriesData.map(e => ({ text: e.name, value: e.id }));
  categoryOptions.push({ text: 'Select a Category', value: '' });
  return (
    <DashboardLayout pageTitle="Product Management">
      <div className="container-fluid">
        <div className="row">
          <div className="col-12">
            <div className="dashCard">
              <div className="row mb-5">
                <div className="col-12 d-flex align-items-center justify-content-between">
                  <h2 className="mainTitle mb-0">Product Management</h2>
                  <CustomButton text="Add Product" onClick={() => navigate('/admin/product-management/add/')} />
                </div>
              </div>
              <div className="statsCard">
                <div className="row mb-3">
                  <div className="col-12">
                    <CustomTable
                      filters={filters}
                      setFilters={setFilters}
                      loading={isSubmitting}
                      headers={productHeaders}
                      pagination={pagination}
                      // if you want multiple date filters
                      dateFilters={[
                        {
                          from: "fromDate",
                          to: "toDate",
                        },
                      ]}
                      // if you want multiple select filters
                      selectOptions={[
                        {
                          title: "status",
                          options: ProductStatus,
                        },
                        {
                          title: "Category",
                          options: categoryOptions,
                        },
                      ]}
                    >
                      <tbody>
                        {productData?.map((item, index) => (
                          <tr key={item?.id}>
                            <td>
                              {serialNum(
                                (filters.page - 1) * filters.per_page + index + 1
                              )}
                            </td>
                            <td>{item?.name}</td>
                            <td>{dateFormat(item?.created_at)}</td>
                            <td>{item?.category?.name}</td>
                            <td
                              className={getStatusByText(item?.status).className}
                            >
                              {getStatusByText(item?.status).text}
                            </td>
                            <td>
                              <TableDropdown
                                view
                                itemId={item?.id}
                                linkPath="/admin/product-management/details"
                                handleStatusChange={confirmPopup}
                                statusDetail={item?.status}
                              />
                            </td>
                          </tr>
                        ))}
                      </tbody>
                    </CustomTable>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </DashboardLayout>
  );
};

//Additional Filters
const additionalFilters = {
  type: "",
  expiryFromDate: "",
  expiryToDate: "",
};

export default withModal(withFilters(ProductManagement, additionalFilters));
// Bind with HOC
// withModal is for Pop Up
// withFilters is for Pagination and default filters
// withPagination, if you want only pagination not filters
// additionalFilters, if you want extra filters on specific pages
