import { useState, useEffect } from "react";
import { useNavigate } from "react-router";
import "./style.css";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { faCamera } from "@fortawesome/free-solid-svg-icons";
import { DashboardLayout } from "../../../Components/AdminLayout/DashboardLayout";
import CustomInput from "../../../Components/CustomInput";
import CustomButton from "../../../Components/CustomButton";
import BackButton from "../../../Components/BackButton";
import { useDispatch } from "react-redux";
import { Formik } from "formik";
import { adminProfileValidation } from "../../../Config/Validations";
import { post } from "../../../Services/Api";
import { setData } from "../../../Store/actions";
import { useFormStatus } from "../../../Hooks/useFormStatus";
import { images } from "../../../Assets";
import { useAuth } from "../../../Hooks/useAuth";
import { validateImage } from "../../../Utils/helper";
import withModal from "../../../HOC/withModal";

const EditProfile = ({ showModal }) => {
  const navigate = useNavigate();
  const [imageObject, setImageObject] = useState({});
  const [profileData, setProfileData] = useState({});
  const [profilePic, setProfilePic] = useState();
  const { isSubmitting, startSubmitting, stopSubmitting } = useFormStatus();
  const dispatch = useDispatch();
  const { user } = useAuth();

  // Fetch user data and set profile info
  useEffect(() => {
    setProfileData(user);
    setProfilePic(user?.image_url);
  }, [user]);

  // Handle image change and validation
  const handleImageChange = (e, setFieldError) => {
    const file = e.target.files[0];
    const errorMsg = validateImage(file);

    if (errorMsg) {
      setFieldError("image", errorMsg);
    } else {
      setFieldError("image", "");
      setProfilePic(URL.createObjectURL(file));
      setImageObject(file);
    }
  };

  // Handle form submission
  const handleSubmit = async (values, { setSubmitting }) => {
    setSubmitting(true);
    console.log("values", values);

    // Create FormData object for file upload
    // const formData = new FormData();
    // formData.append("first_name", values.first_name);
    // formData.append("last_name", values.last_name);

    // If an image was selected, append it to the form data
    if (imageObject) {
      values.image = imageObject;
      // formData.append("image", imageObject);
    }
    const response = await post("/admin/profile/update", values);
    if (response.status) {
      setSubmitting(false);
      showModal(`Profile updated successfully`, null, true, "/admin/profile");
      dispatch(setData(response.data));
    }
    setSubmitting(false);

  };

  return (
    <>
      <DashboardLayout pageTitle="Edit Profile">
        <div className="dashCard mb-4">
          <div className="row mb-3">
            <div className="col-12">
              <h2 className="mainTitle">
                <BackButton />
                Edit Profile
              </h2>
            </div>
          </div>

          <div className="row mb-3">
            {profileData ? (
              <div className="col-lg-8">
                {Object.keys(profileData).length > 0 && (
                  <Formik
                    initialValues={{
                      first_name: profileData.first_name || "",
                      last_name: profileData.last_name || "",
                    }}
                    validationSchema={adminProfileValidation}
                    onSubmit={handleSubmit}
                  >
                    {({
                      values,
                      errors,
                      touched,
                      handleChange,
                      handleBlur,
                      handleSubmit,
                      setFieldValue,
                      setFieldError,
                    }) => (
                      <form onSubmit={handleSubmit}>
                        <div className="row mb-3">
                          <div className="col-lg-4 mb-3">
                            <div className="profileImage">
                              <img
                                src={profilePic ?? images.userImage}
                                alt="User"
                              />
                              <input
                                type="file"
                                accept="image/*"
                                className="d-none"
                                id="profileImage"
                                onChange={(event) =>
                                  handleImageChange(event, setFieldError)
                                }
                              />
                              <label
                                htmlFor="profileImage"
                                className="imageUploadButton"
                              >
                                <FontAwesomeIcon icon={faCamera} />
                              </label>
                            </div>
                            {errors.image && (
                              <div className="errorText red-text">
                                {errors.image}
                              </div>
                            )}
                          </div>
                        </div>

                        <div className="row mb-3">
                          <div className="col-md-8">
                            <CustomInput
                              label="First Name"
                              labelclass="mainLabel"
                              type="text"
                              placeholder="Enter First Name"
                              inputclass="mainInput"
                              id="first_name"
                              value={values.first_name}
                              onChange={handleChange}
                              onBlur={handleBlur}
                              error={
                                touched.first_name && errors.first_name
                              }
                            />
                          </div>
                        </div>

                        <div className="row mb-3">
                          <div className="col-md-8">
                            <CustomInput
                              label="Last Name"
                              labelclass="mainLabel"
                              type="text"
                              placeholder="Enter Last Name"
                              inputclass="mainInput"
                              id="last_name"
                              value={values.last_name}
                              onChange={handleChange}
                              onBlur={handleBlur}
                              error={touched.last_name && errors.last_name}
                            />
                          </div>
                        </div>

                        <div className="row">
                          <div className="col-md-4">
                            <CustomButton
                              variant="btnBlueBG"
                              className="px-5"
                              text="Update"
                              pendingText="Submitting..."
                              isPending={isSubmitting}
                              type="submit"
                            />
                          </div>
                        </div>
                      </form>
                    )}
                  </Formik>
                )}
              </div>
            ) : (
              ""
            )}
          </div>
        </div>
      </DashboardLayout>
    </>
  );
};

export default withModal(EditProfile);
