import { useState, useEffect } from "react";
import { DashboardLayout } from "../../../Components/AdminLayout/DashboardLayout";
import CustomTable from "../../../Components/CustomTable";
import { dateFormat, getStatus, serialNum } from "../../../Utils/helper";
import { userHeaders } from "../../../Config/TableHeaders"; // Table Headers
import { generalStatus } from "../../../Config/TableStatus"; // Filter Status
import { getAll, post } from "../../../Services/Api"; //Api Service
import TableDropdown from "../../../Components/TableDropdown";
import "./style.css";
import { useFormStatus } from "../../../Hooks/useFormStatus";
import withFilters from "../../../HOC/withFilters ";
import withModal from "../../../HOC/withModal";
import { userManagementData } from "../../../Config/data";

const UserManagement = ({
  showModal,
  filters,
  setFilters,
  pagination,
  updatePagination,
}) => {
  const [userData, setUserData] = useState([]);
  const { isSubmitting, startSubmitting, stopSubmitting } = useFormStatus();

  const fetchUsers = async () => {
    startSubmitting(true);
    const url = `/admin/users`;
    const response = await getAll(url, filters);
    if (response.status) {
      const { total, per_page, current_page, to } = response.meta;
      setUserData(response.data);
      updatePagination({
        showData: to,
        currentPage: current_page,
        totalRecords: total,
        totalPages: Math.ceil(total / per_page),
      });
      stopSubmitting(false);

    }
    stopSubmitting(false);

  };

  const confirmPopup = (id, status) => {
    showModal(
      `Are you sure you want to ${status === 1 ? "Inactivate" : "Activate"
      } this User?`,
      () => onConfirm(id, status)
    );
  };

  const onConfirm = async (id, status) => {
    const response = await post(`/admin/user/${id}/status`);
    if (response.status) {
      await fetchUsers();
      showModal(
        `User ${response?.data?.status === 0 ? "inactivated" : "activated"
        } successfully`, //heading
        null, //action
        true //sucess
      );
    }
  };

  useEffect(() => {
    fetchUsers();
  }, [filters]);

  return (
    <DashboardLayout pageTitle="User Management">
      <div className="container-fluid">
        <div className="row">
          <div className="col-12">
            <div className="dashCard">
              <div className="row mb-5">
                <div className="col-md-6">
                  <h2 className="mainTitle mb-0">Users Management</h2>
                </div>
              </div>
              <div className="statsCard">
                <div className="row mb-3">
                  <div className="col-12">
                    <CustomTable
                      filters={filters}
                      setFilters={setFilters}
                      loading={isSubmitting}
                      headers={userHeaders}
                      pagination={pagination}
                      // if you want multiple date filters
                      dateFilters={[
                        {
                          from: "from",
                          to: "to",
                        },
                      ]}
                      // if you want multiple select filters
                      selectOptions={[
                        {
                          title: "status",
                          options: generalStatus,
                        },
                      ]}
                    >
                      <tbody>
                        {userData?.map((item, index) => (
                          <tr key={item?.id}>
                            <td>
                              {serialNum(
                                (filters?.page - 1) * filters?.per_page + index + 1
                              )}
                            </td>
                            <td>{item?.full_name}</td>
                            <td>{item?.email}</td>
                            <td>{dateFormat(item?.created_at)}</td>

                            <td
                              className={item?.is_verified == 1 ? "greenColor" : "redColor"}
                            >
                              {item?.is_verified == 1 ? "Verfied" : "Not-Verfied"}
                            </td>
                            <td
                              className={item?.status == 1 ? "greenColor" : "redColor"}
                            >
                              {item?.status == 1 ? "Active" : "In-Active"}
                            </td>
                            <td>
                              <TableDropdown
                                view
                                itemId={item?.id}
                                linkPath="/admin/user-management"
                                handleStatusChange={() => confirmPopup(item?.id, item?.status)}
                                statusDetail={item?.status === 1 ? "active" : "inactive"}
                              />
                            </td>
                          </tr>
                        ))}
                      </tbody>
                    </CustomTable>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </DashboardLayout>
  );
};

//Additional Filters
const additionalFilters = {
  type: "",
  expiryFrom: "",
  expiryTo: "",
};

export default withModal(withFilters(UserManagement, additionalFilters));
// Bind with HOC
// withModal is for Pop Up
// withFilters is for Pagination and default filters
// withPagination, if you want only pagination not filters
// additionalFilters, if you want extra filters on specific pages
