import React, { useEffect, useState } from 'react';
import ReviewCard from '../../../Components/ReviewCard';
import { ButtonGroup, ProgressBar, ToggleButton } from 'react-bootstrap';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { faStar } from '@fortawesome/free-solid-svg-icons';
import { UserLayout } from '../../../Components/user/Layout';
import { useNavigate, useParams } from 'react-router-dom';
import { userProducts } from '../../../Config/userData';
import BackButton from '../../../Components/BackButton';
import QuantityInput from '../../../Components/user/QuantityInput/QuantityInput';
import CustomButton from '../../../Components/CustomButton';
import { faHeart } from '@fortawesome/free-regular-svg-icons';
import { faHeart as filledHeart } from '@fortawesome/free-solid-svg-icons';
import UserReviewCard from '../../../Components/user/ReviewCard';
import { deleteData, getDetails, post } from '../../../Services/Api';
import LoadingSpinner from '../../../Components/Loader';
import { useDispatch } from 'react-redux';
import { addToCart } from '../../../Store/actions/cartActions';
import { ImageSliderMultiple } from '../../../Components/ImageSliderMultiple';
import { SiteSlider } from '../../../Components/SiteSlider';
import ProductCard from '../ProductCard/ProductCard';
import './index.css';
import { useAuth } from '../../../Hooks/useAuth';
import CustomModal from "../../../Components/CustomModal";

const UserProductDetails = () => {
    const { id } = useParams();
    const [details, setDetails] = useState({});
    const [recentlyViewedProducts, setRecentlyViewedProducts] = useState([]);
    const [loadingProduct, setLoadingProduct] = useState(false);
    const [loadingRecentlyViewed, setLoadingRecentlyViewed] = useState(false);
    const [quantity, setQuantity] = useState(1);
    const [wishlistLoading, setWishlistLoading] = useState(false);
    const [showModal, setShowModal] = useState(false);

    const [filled, setFilled] = useState(false);
    const [radioValue, setRadioValue] = useState(null);
    const { user, sessionId } = useAuth();
    const navigate = useNavigate();
    const dispatch = useDispatch();

    const getProduct = async () => {
        try {
            setLoadingProduct(true);
            const response = await getDetails(`/user/product/${id}`);
            if (response.status) {
                setDetails(response.data);
                setRadioValue(response.data?.product_sizes[0]?.id); // Default to the first size
            }
        } catch (error) {
            console.error("Error fetching product details:", error);
        } finally {
            setLoadingProduct(false);
        }
    };

    const getRecentlyViewedProducts = async () => {
        try {
            setLoadingRecentlyViewed(true);
            const response = await getDetails(`/user/recentlyViewedProduct/recentlyViewedProducts`);
            if (response.status) {
                setRecentlyViewedProducts(response.data);
                handleAddToRecentlyViewedProduct(); // Add current product to recently viewed
            }
        } catch (error) {
            console.error("Error fetching recently viewed products:", error);
        } finally {
            setLoadingRecentlyViewed(false);
        }
    };

    const addProductToWishlist = async (productId, isInWishList, wishlist) => {
        try {
            setWishlistLoading(true);
            let response;
            if (!isInWishList) {
                response = await post('/user/wishlist/add', { product_id: productId });
            } else {
                response = await deleteData(`/user/wishlist/remove/${wishlist?.id}`);
            }
            if (response.status) {
                getProduct(); // Refresh product details after wishlist change
            }
        } catch (error) {
            console.error("Error updating wishlist:", error);
        } finally {
            setWishlistLoading(false);
        }
    };

    const handleAddToCart = async (id) => {
        try {
            let user_id;
            if (!user) {
                user_id = sessionId.sessionId;
                // If user is not logged in, redirect to the login page
                // navigate("/login");
                // return;
            } else {
                user_id = user.id;
            }
            const response = await post('user/cart/add', { 'product_id': id, 'quantity': quantity, 'size_id': radioValue, 'user_id': user_id });
            if (response.status) {
                console.log("api response", response.data);
                setShowModal(true)
                dispatch(addToCart({ 'id': id, }));

            }
        } catch (error) {
            console.error("Error adding to cart:", error);
        }
    };

    const handleAddToRecentlyViewedProduct = async () => {
        try {
            await post('user/recentlyViewedProduct/add', { 'product_id': id });
        } catch (error) {
            console.error("Error adding product to recently viewed:", error);
        }
    };

    const handleQuantityChange = (newQuantity) => {
        setQuantity(newQuantity);
    };

    const productLiked = () => setFilled(!filled);

    useEffect(() => {
        getProduct();
        if (user) {
            getRecentlyViewedProducts();
        }

    }, [id]);

    return (
        <>
            <UserLayout>
                <section className='product-details py-sm-5 py-4'>
                    <div className="container">
                        <div className="row">
                            <div className="col-12">
                                <div className="mt-4">
                                    <BackButton />
                                    {loadingProduct || loadingRecentlyViewed ? (
                                        <LoadingSpinner />
                                    ) : (
                                        <>
                                            <div className="row">
                                                <div className="col-lg-6 pe-lg-5 my-3">
                                                    <div className="position-relative">
                                                        <ImageSliderMultiple
                                                            className="mxw-100"
                                                            productDetailsClass="product-detail-slider-img"
                                                            thumbnailClass="product-thumbnail-img"
                                                            images={details?.images}
                                                        />
                                                        {/* <button
                                                        className="transparent-btn heart-icon"
                                                        onClick={() => addProductToWishlist(details?.id, details?.is_in_wishlist, details?.wishlist)}
                                                        disabled={wishlistLoading}
                                                    >
                                                        <FontAwesomeIcon
                                                            icon={details?.is_in_wishlist ? filledHeart : faHeart}
                                                            className='redColor'
                                                        />
                                                    </button> */}
                                                    </div>
                                                </div>
                                                <div className="col-lg-6 my-3">
                                                    <div className="d-flex align-items-center mt-4 gap-3 justify-content-between flex-wrap-reverse">
                                                        <div>
                                                            <h4 className="mb-0 bold gilroy text-uppercase">{details?.name}</h4>
                                                            <p className='mb-0'>SKU: {details?.sku}</p>
                                                        </div>
                                                        <div>
                                                            <p className="p-md mb-0 bold gilroy text-uppercase">${details?.price - details?.discounted_price}</p>
                                                            <p className='mb-0'><del>${details?.price}</del></p>
                                                        </div>
                                                    </div>
                                                    <p className="mb-0 mt-3">{details?.description}</p>
                                                    <div className="d-flex align-items-center mt-4 gap-3 flex-wrap">
                                                        <CustomButton text="Add To Cart" className="py-2" onClick={() => handleAddToCart(details?.id)} />
                                                    </div>
                                                    {
                                                        details?.product_sizes?.[0]?.name !== "" && (
                                                            <>
                                                                <p className="mb-0 mt-3">Sizes:</p>
                                                                <ButtonGroup className="mb-2 site-radio-btns">
                                                                    {details?.product_sizes?.map((radio, idx) => (
                                                                        <ToggleButton
                                                                            key={idx}
                                                                            id={`radio-${idx}`}
                                                                            type="radio"
                                                                            variant="secondary"
                                                                            name="radio"
                                                                            value={radio?.id}
                                                                            checked={radioValue === radio.id}
                                                                            onChange={() => setRadioValue(radio.id)} // Updated event handler
                                                                        >
                                                                            {radio?.name}
                                                                        </ToggleButton>
                                                                    ))}
                                                                </ButtonGroup>
                                                            </>
                                                        )
                                                    }
                                                </div>
                                            </div>
                                            {recentlyViewedProducts.length > 0 && (
                                                <div className="mt-4">
                                                    <h4 className="bold mb-3">Recently Viewed Products</h4>
                                                    <div className="row">
                                                        {recentlyViewedProducts.length <= 3 ? (
                                                            recentlyViewedProducts.map((item) => (
                                                                <div className='col-md-6 col-lg-4' key={item.id}>
                                                                    <ProductCard handleAddWishlist={getRecentlyViewedProducts} {...item?.product} />
                                                                </div>
                                                            ))
                                                        ) : (
                                                            <div className='col-md-12'>
                                                                <SiteSlider>
                                                                    {recentlyViewedProducts.map((item) => (
                                                                        <div className='px-2' key={item.id}>
                                                                            <ProductCard handleAddWishlist={getRecentlyViewedProducts} {...item?.product} />
                                                                        </div>
                                                                    ))}
                                                                </SiteSlider>
                                                            </div>
                                                        )}
                                                    </div>
                                                </div>
                                            )}
                                        </>
                                    )}
                                </div>
                            </div>
                        </div>
                    </div>
                </section>
            </UserLayout>
            <CustomModal
                show={showModal}
                close={() => setShowModal(false)}
                success
                heading="Product Has been Added to the Cart"
                para=""
            />
        </>

    );
};

export default UserProductDetails;
