import { useEffect, useState, useCallback } from 'react';
import { UserLayout } from '../../../Components/user/Layout';
import InnerBanner from '../../../Components/user/InnerBanner/InnerBanner';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { faFilter, faSearch } from '@fortawesome/free-solid-svg-icons';
import SiteInput from '../../../Components/Input/input';
import { Dropdown, Form } from 'react-bootstrap';
import { getAll } from '../../../Services/Api';
import LoadingSpinner from '../../../Components/Loader';
import ProductCard from '../ProductCard/ProductCard';
import withFilters from "../../../HOC/withFilters ";
import CustomPagination from "../../../Components/CustomPagination";
import debounce from "lodash.debounce";
import { useAuth } from '../../../Hooks/useAuth';
import { useParams } from 'react-router-dom';
import BackButton from '../../../Components/BackButton';

const Products = ({ filters, setFilters, pagination, updatePagination }) => {
    const [productData, setProductData] = useState([]);
    const { id } = useParams();
    const [categoriesData, setCategoriesData] = useState([]);
    const [loading, setLoading] = useState(true);
    const [formData, setFormData] = useState({ search: "" });
    const [selectedCategories, setSelectedCategories] = useState([]);
    const { user } = useAuth();

    const fetchData = async () => {
        setLoading(true);
        var url
        if (user) {
            url = `/user/product/products/${id}`;
        } else {
            url = `user/product/products/unathenticated/${id}`;
        }
        const response = await getAll(url, { ...filters, categories: selectedCategories });
        if (response.status) {
            const { total, per_page, current_page, to } = response.meta;
            setProductData(response.data);
            updatePagination({
                showData: to,
                currentPage: current_page,
                totalRecords: total,
                totalPages: Math.ceil(total / per_page),
            });
        }
        setLoading(false);
    };

    const fetchCategoriesData = async () => {
        setLoading(true);
        const response = await getAll(`/user/category/all`);
        console.log("API response:", response);

        if (response.status) {
            setCategoriesData(response.data);
        }
        setLoading(false);
    };

    const debouncedSetFilters = useCallback(
        debounce((updatedFormData) => {
            setFilters(updatedFormData);
        }, 500),
        [setFilters]
    );

    const handleChange = (name, value) => {
        const updatedFormData = { ...formData, [name]: value };
        setFormData(updatedFormData);

        if (name === "search") {
            debouncedSetFilters(updatedFormData);
        } else if (name === "per_page") {
            setFilters(updatedFormData);
        }
    };

    const handleInputChange = (event) => {
        const { name, value } = event.target;
        handleChange(name, value);
    };

    const handleCheckboxChange = (event) => {
        const categoryId = parseInt(event.target.id, 10);
        setSelectedCategories((prev) => {
            const updatedCategories = event.target.checked
                ? [...prev, categoryId]
                : prev.filter(id => id !== categoryId);

            setFilters((prevFilters) => ({
                ...prevFilters,
                categories: updatedCategories,
            }));
            return updatedCategories;
        });
    };

    useEffect(() => {
        fetchCategoriesData();
        fetchData();
    }, []); // Fetch categories and initial data on mount

    useEffect(() => {
        fetchData(); // Fetch data when selectedCategories change
    }, [selectedCategories, filters]);

    return (
        <UserLayout>
            <InnerBanner title={'Bands'} className="about-banner about-banner-sm" />
            <section className="about-us py-sm-5 py-4">
                <div className="container">
                    <div className="row">
                        <div className="col-12">
                            <BackButton />

                            <div className="d-flex align-items-center gap-3 justify-content-end">
                                <div className="search-barr">
                                    <FontAwesomeIcon icon={faSearch} className="left-icon" />
                                    <SiteInput
                                        name="search"
                                        type="text"
                                        className="mainInput ps-5"
                                        value={formData.search}
                                        onChange={handleInputChange}
                                        placeholder="Search"
                                    />
                                </div>
                                {/* <Dropdown>
                                    <Dropdown.Toggle variant="success" className='transparent-btn' id="dropdown-basic">
                                        <FontAwesomeIcon icon={faFilter} className='blackColor' />
                                    </Dropdown.Toggle>
                                    <Dropdown.Menu align={'end'} className='p-3'>
                                        <Form>

                                            {categoriesData.map((item) => (
                                                <Form.Check
                                                    type='checkbox'
                                                    id={item.id}
                                                    label={item.name}
                                                    key={item.id}
                                                    onChange={handleCheckboxChange}
                                                    checked={selectedCategories.includes(item.id)}
                                                />
                                            ))}
                                        </Form>
                                    </Dropdown.Menu>
                                </Dropdown> */}
                            </div>
                        </div>
                        {loading ? (
                            <LoadingSpinner />
                        ) : productData.length > 0 ? (
                            productData.map((item) => (
                                <div className="col-lg-4 my-3 col-md-6" key={item.id}>
                                    <ProductCard handleAddWishlist={fetchData} {...item} />
                                </div>
                            ))
                        ) : (
                            <div>
                                <p>Data not found</p>
                            </div>
                        )}
                        {pagination && (
                            <CustomPagination
                                pagination={pagination}
                                setFilters={setFilters}
                            />
                        )}
                    </div>
                </div>
            </section>
        </UserLayout>
    );
};

export default withFilters(Products);
